# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for the :class:`~anaconda_navigator.utils.attribution.pool.Updater`."""

from __future__ import annotations

__all__ = ()

import os
import typing

import pytest
import yaml


def test_broken_configuration() -> None:
    """Check how component works with broken partner configuration files."""
    from anaconda_navigator.config import preferences
    from anaconda_navigator.utils import attribution

    path: str = os.path.join(os.path.expanduser('~'), 'test_partner.yml')

    stream: typing.TextIO
    with open(path, 'wt', encoding='utf-8') as stream:
        stream.write(':')

    preferences.AD_CONFIGURATION_PATHS = [path]  # type: ignore

    assert attribution.UPDATER.instance.partner_configuration == {'utm_campaign': ''}


def test_empty_configuration() -> None:
    """Check how component works with non-existing partner configuration files."""
    from anaconda_navigator.config import preferences
    from anaconda_navigator.utils import attribution

    preferences.AD_CONFIGURATION_PATHS = [os.path.join(os.path.expanduser('~'), 'test_partner.yml')]  # type: ignore

    assert attribution.UPDATER.instance.partner_configuration == {'utm_campaign': ''}


@pytest.mark.parametrize('content', (
    pytest.param({}, id='empty'),
    pytest.param({'unexpected_field': ''}, id='invalid'),
    pytest.param({'utm_campaign': ''}, id='empty_campaign'),
    pytest.param({'utm_campaign': '!'}, id='minimal_campaign'),
    pytest.param({'utm_campaign': 'value'}, id='generic_campaign'),
    pytest.param({'utm_campaign': '01e1596d-9850-4d36-8e33-a7a43f93cfda'}, id='uuid_campaign'),
    pytest.param({'utm_campaign': 'a' * 64}, id='long_campaign'),
))
def test_existing_configuration(content: typing.Any) -> None:
    """Check how component works with existing valid partner configurations."""
    from anaconda_navigator.config import preferences
    from anaconda_navigator.utils import attribution

    path: str = os.path.join(os.path.expanduser('~'), 'test_partner.yml')

    stream: typing.TextIO
    with open(path, 'wt', encoding='utf-8') as stream:
        yaml.dump(content, stream)

    preferences.AD_CONFIGURATION_PATHS = [path]  # type: ignore

    content.setdefault('utm_campaign', '')
    assert attribution.UPDATER.instance.partner_configuration == content


@pytest.mark.parametrize('content', (
    pytest.param({'utm_campaign': 'a' * 65}, id='long_campaign'),
    pytest.param({'utm_campaign': None}, id='null_campaign'),
    pytest.param({'utm_campaign': True}, id='bool_campaign'),
    pytest.param({'utm_campaign': {}}, id='object_campaign'),
    pytest.param({'utm_campaign': []}, id='array_campaign'),
))
def test_utm_campaign_reset(content: typing.Any) -> None:
    """Check how component resets invalid `utm_campaign` values."""
    from anaconda_navigator.config import preferences
    from anaconda_navigator.utils import attribution

    path: str = os.path.join(os.path.expanduser('~'), 'test_partner.yml')

    stream: typing.TextIO
    with open(path, 'wt', encoding='utf-8') as stream:
        yaml.dump(content, stream)

    preferences.AD_CONFIGURATION_PATHS = [path]  # type: ignore

    content['utm_campaign'] = ''
    assert attribution.UPDATER.instance.partner_configuration == content
