# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring,protected-access

"""Tests for CloudAPI."""

from __future__ import annotations

__all__ = ()

import contextlib
import io
import os
import tempfile
import typing
import uuid
from unittest import mock

from anaconda_auth import client

TEST_USER = 'test_user'
TEST_PASSWORD = 'test_password'  # nosec

TEST_ACCESS_TOKEN = 'some_access_token'  # nosec
TEST_REFRESH_TOKEN = 'some_refresh_token'  # nosec
TEST_TOKEN_TYPE = 'Bearer'  # nosec
TEST_EXPIRES_IN = 900

AUTH_RESPONSE_DATA = {
    'access_token': TEST_ACCESS_TOKEN,
    'expires_in': TEST_EXPIRES_IN,
    'refresh_token': 'HttpOnly',
    'token_type': TEST_TOKEN_TYPE
}

AUTH_RESPONSE_COOKIES = {
    'refresh_token': TEST_REFRESH_TOKEN
}


class SemiLock:  # pylint: disable=missing-class-docstring

    def __init__(self):
        """Initialize new :class:`~SemiLock` instance."""

    def __enter__(self) -> None:
        """Context entry."""
        return None

    def __exit__(self, exc_type, value, traceback):
        """Context exit."""


class CommonPatches:
    """Collection of mocks used for patching CloudAPI components."""

    __slots__ = ('token', 'anaconda_auth', 'session')

    def __init__(self) -> None:
        """Initialize new :class:`~CommonPatches` instance."""
        from anaconda_navigator.api.cloud.utilities import sessions

        self.anaconda_auth: typing.Final[typing.Any] = mock.Mock()
        self.token = f'T_ACCESS_TOKEN_{uuid.uuid4().hex}'

        self.session: typing.Final[typing.Any] = mock.Mock()
        self.session.return_value.cookies = sessions.SessionOptions(
            lock=SemiLock(),  # type: ignore
            content={},
        )
        self.session.return_value.headers = sessions.SessionOptions(
            lock=SemiLock(),  # type: ignore
            content={},
        )
        self.update_token()

    @property
    def session_value(self) -> typing.Any:  # noqa: D401
        """Value of the created session."""
        return self.session.return_value

    @property
    def session_response(self) -> typing.Any:  # noqa: D401
        """Response of the session request."""
        return self.session_value.request.return_value

    @property
    def authorization(self):
        return f"Bearer {self.token or ''}"

    def update_token(self):
        token_info = client.TokenInfo(
            api_key=self.token,
            username=None,
        )
        self.anaconda_auth.client.TokenInfo.load.return_value = token_info


@contextlib.contextmanager
def common_patches() -> typing.Iterator[CommonPatches]:
    """Create a controllable environment for test."""
    result = CommonPatches()
    with \
            mock.patch('anaconda_navigator.api.cloud.api.anaconda_auth', result.anaconda_auth), \
            mock.patch.object(client.TokenInfo, 'expired', new_callable=mock.PropertyMock, return_value=False), \
            mock.patch('anaconda_navigator.api.cloud.api.utilities.Session', result.session):
        yield result


def test_init():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        # check initial token value
        assert instance.token is patches.token

        # check session initializations
        patches.session.assert_called_with()
        assert patches.session_value.headers['Authorization'] == patches.authorization


def test_login():
    from anaconda_navigator.api.cloud import api as cloud_api
    from anaconda_navigator.config import CONF

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()
        assert instance.token is patches.token
        new_token: typing.Any = TEST_ACCESS_TOKEN
        patches.token = new_token
        patches.update_token()

        # pylint: disable=no-value-for-parameter
        instance.login()
        patches.session_value.request.assert_called()

        ssl_verify = CONF.get('main', 'ssl_verification', True)
        patches.anaconda_auth.login.assert_called_with(ssl_verify=ssl_verify)
        assert instance.token is new_token
        assert patches.session_value.headers['Authorization'] == patches.authorization


def test_logout():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        # new_token: typing.Any = patches.create_token()
        patches.token = None

        instance.logout()
        assert instance.token is None
        patches.anaconda_auth.logout.assert_called_with()
        assert patches.session_value.headers['Authorization'] == patches.authorization
        patches.session_value.request.assert_called()


def test_list_environments():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        assert instance.list_environments() is patches.session_response.json.return_value
        patches.session_value.request.assert_called()


def test_create_environment():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        temp_stream: typing.TextIO
        temp_path: str = os.path.join(tempfile.gettempdir(), uuid.uuid4().hex)
        with open(temp_path, 'wt', encoding='utf-8') as temp_stream:
            temp_stream.write(''.join(uuid.uuid4().hex for _ in range(128)))

        instance.create_environment(name=uuid.uuid4().hex, path=temp_path)
        patches.session_value.request.assert_called()

        os.remove(temp_path)


def test_update_environment():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        temp_stream: typing.TextIO
        temp_path: str = os.path.join(tempfile.gettempdir(), uuid.uuid4().hex)
        with open(temp_path, 'wt', encoding='utf-8') as temp_stream:
            temp_stream.write(''.join(uuid.uuid4().hex for _ in range(64)))

        instance.update_environment(name=uuid.uuid4().hex, path=temp_path, rename_to=uuid.uuid4().hex)
        patches.session_value.request.assert_called()

        os.remove(temp_path)


def test_delete_environment():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        instance.delete_environment(name=uuid.uuid4().hex)
        patches.session_value.request.assert_called()


def test_download_environment():
    from anaconda_navigator.api.cloud import api as cloud_api

    patches: CommonPatches
    with common_patches() as patches:
        instance: cloud_api._CloudAPI = cloud_api._CloudAPI()

        temp_path: str = os.path.join(tempfile.gettempdir(), uuid.uuid4().hex)
        content: bytes = b''.join(uuid.uuid4().bytes for _ in range(32))
        stream: typing.BinaryIO = io.BytesIO(content)
        patches.session_response.raw = stream

        instance.download_environment(name=uuid.uuid4().hex, path=temp_path)
        patches.session_value.request.assert_called()

        temp_stream: typing.BinaryIO
        with open(temp_path, 'rb') as temp_stream:
            assert temp_stream.read() == content

        os.remove(temp_path)
