# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring,redefined-outer-name

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for quit-like dialogs."""

import typing
from qtpy.QtCore import Qt
import pytest
from tests.current.widgets import common_extra


class RunningProcess(typing.NamedTuple):
    """Mock for running processes."""

    package: str
    pid: int


RUNNING_PROCESSES = [
    RunningProcess(package='test-app-1', pid=12),
    RunningProcess(package='test-app-2', pid=345),
    RunningProcess(package='test-app-3', pid=678),
]
APP_IN_LIST = RUNNING_PROCESSES[0].package


# --- Fixtures
# -----------------------------------------------------------------------------
@pytest.fixture(scope='function')
def botquittrue(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator import config as navigator_config
    from anaconda_navigator.widgets.dialogs import quit as quit_dialogs

    navigator_config.CONF.set('main', 'hide_quit_dialog', True)

    widget = quit_dialogs.QuitApplicationDialog()
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botquitfalse(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator import config as navigator_config
    from anaconda_navigator.widgets.dialogs import quit as quit_dialogs

    navigator_config.CONF.set('main', 'hide_quit_dialog', False)

    widget = quit_dialogs.QuitApplicationDialog()
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botquitbusy(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator.widgets.dialogs import quit as quit_dialogs

    widget = quit_dialogs.QuitBusyDialog()
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def closecondawidget(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator.widgets.dialogs import quit as quit_dialogs

    widget = quit_dialogs.ClosePackageManagerDialog()
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botquitrunningappshide(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator import config as navigator_config
    from anaconda_navigator.widgets.dialogs import quit as quit_dialogs

    navigator_config.CONF.set('main', 'running_apps_to_close', [APP_IN_LIST])
    navigator_config.CONF.set('main', 'hide_running_apps_dialog', True)

    widget = quit_dialogs.QuitRunningAppsDialog(running_processes=RUNNING_PROCESSES)
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botquitrunningappsshow(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator import config as navigator_config
    from anaconda_navigator.widgets.dialogs import quit as quit_dialogs

    navigator_config.CONF.set('main', 'running_apps_to_close', [APP_IN_LIST])
    navigator_config.CONF.set('main', 'hide_running_apps_dialog', False)

    widget = quit_dialogs.QuitRunningAppsDialog(running_processes=RUNNING_PROCESSES)
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


def test_quit_dialog_hide(botquitfalse):
    assert botquitfalse.widget.checkbox.checkState() == Qt.Unchecked


def test_quit_dialog_show(botquittrue):
    assert botquittrue.widget.checkbox.checkState() == Qt.Checked


def test_quit_dialog_accept(botquitfalse):
    botquitfalse.qtbot.mouseClick(botquitfalse.widget.checkbox, Qt.LeftButton)
    with botquitfalse.qtbot.waitSignal(botquitfalse.widget.accepted, timeout=1000, raising=True):
        botquitfalse.qtbot.mouseClick(botquitfalse.widget.button_ok, Qt.LeftButton)
    assert botquitfalse.widget.config.get('main', 'hide_quit_dialog')


def test_quit_dialog_reject(botquittrue):
    botquittrue.qtbot.mouseClick(botquittrue.widget.checkbox, Qt.LeftButton)
    with botquittrue.qtbot.waitSignal(botquittrue.widget.rejected, timeout=1000, raising=True):
        botquittrue.qtbot.mouseClick(botquittrue.widget.button_cancel, Qt.LeftButton)
    assert botquittrue.widget.config.get('main', 'hide_quit_dialog')


def test_quit_busy_dialog_accept(botquitbusy):
    with botquitbusy.qtbot.waitSignal(botquitbusy.widget.accepted, timeout=1000, raising=True):
        botquitbusy.qtbot.mouseClick(botquitbusy.widget.button_ok, Qt.LeftButton)


def test_quit_busy_dialog_reject(botquitbusy):
    with botquitbusy.qtbot.waitSignal(botquitbusy.widget.rejected, timeout=1000, raising=True):
        botquitbusy.qtbot.mouseClick(botquitbusy.widget.button_cancel, Qt.LeftButton)


def test_close_conda_dialog_ok(closecondawidget):
    with closecondawidget.qtbot.waitSignal(closecondawidget.widget.accepted, timeout=1000, raising=True):
        closecondawidget.qtbot.mouseClick(closecondawidget.widget.button_ok, Qt.LeftButton)


def test_close_conda_dialog_cancel(closecondawidget):
    with closecondawidget.qtbot.waitSignal(closecondawidget.widget.rejected, timeout=1000, raising=True):
        closecondawidget.qtbot.mouseClick(closecondawidget.widget.button_cancel, Qt.LeftButton)


def test_quit_running_apps_dialog_hide(botquitrunningappshide):
    assert botquitrunningappshide.widget.checkbox.checkState() == Qt.Checked


def test_quit_running_apps_dialog_show(botquitrunningappsshow):
    assert botquitrunningappsshow.widget.checkbox.checkState() == Qt.Unchecked


def test_quit_running_apps_dialog_accept(botquitrunningappsshow):
    from anaconda_navigator import config as navigator_config

    item = botquitrunningappsshow.widget.list.item(0)
    item.set_checked(False)
    with botquitrunningappsshow.qtbot.waitSignal(botquitrunningappsshow.widget.accepted, timeout=5000, raising=True):
        botquitrunningappsshow.qtbot.mouseClick(botquitrunningappsshow.widget.button_close, Qt.LeftButton)
    apps = navigator_config.CONF.get('main', 'running_apps_to_close')
    assert apps == []


def test_quit_running_apps_dialog_accept_2(botquitrunningappsshow):
    from anaconda_navigator import config as navigator_config

    item = botquitrunningappsshow.widget.list.item(1)
    item.set_checked(True)
    with botquitrunningappsshow.qtbot.waitSignal(botquitrunningappsshow.widget.accepted, timeout=5000, raising=True):
        botquitrunningappsshow.qtbot.mouseClick(botquitrunningappsshow.widget.button_close, Qt.LeftButton)
    apps = navigator_config.CONF.get('main', 'running_apps_to_close')
    assert apps == ['test-app-1', 'test-app-2']


def test_quit_running_apps_dialog_reject(botquitrunningappsshow):
    from anaconda_navigator import config as navigator_config

    item = botquitrunningappsshow.widget.list.item(2)
    item.set_checked(True)
    with botquitrunningappsshow.qtbot.waitSignal(botquitrunningappsshow.widget.rejected, timeout=5000, raising=True):
        botquitrunningappsshow.qtbot.mouseClick(botquitrunningappsshow.widget.button_cancel, Qt.LeftButton)
    apps = navigator_config.CONF.get('main', 'running_apps_to_close')
    assert apps == ['test-app-1']
