# -*- coding: utf-8 -*-

"""Common configuration for tests."""

from __future__ import annotations

__all__ = ()

import sys
import types
import typing

import pytest
from qtpy import QtCore

from tests.utilities import qt_utils

if typing.TYPE_CHECKING:
    import pytestqt.plugin


QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_UseSoftwareOpenGL)
QtCore.QCoreApplication.setAttribute(QtCore.Qt.AA_ShareOpenGLContexts)


@pytest.fixture(scope='function', autouse=True)
def each_test(isolated_home: None, qt_styles: None) -> typing.Iterator[None]:  # pylint: disable=unused-argument
    """Common fixture for each test."""
    from anaconda_navigator.widgets.dialogs import splash  # pylint: disable=import-outside-toplevel
    from anaconda_navigator.widgets import main_window  # pylint: disable=import-outside-toplevel

    splash_screen: splash.SplashScreen = splash.SplashScreen()
    splash_screen.show_message('Initializing for testing...')

    window: main_window.MainWindow = main_window.MainWindow(splash=splash_screen)

    try:
        yield None
    finally:
        window.destroy(True, True)


@pytest.fixture(scope='function')
def qt_helper(
        qtbot: 'pytestqt.plugin.QtBot',  # pylint: disable=unused-argument
) -> typing.Iterator['qt_utils.QtHelper']:
    """Shortcut for :class:`~tests.utilities.qt_testing.qt_helper.QtHelper` instance."""
    yield qt_utils.QtHelper()


class TestThread(QtCore.QThread):
    """
    Separate thread to run a test body in.

    Main reason for it - is that launching any dialog may block the main thread. If test is launched in the main thread
    - it would become blocked as well and won't be able to proceed.

    On other hand, if test is running in a separate thread - it should not become blocked. It also should have access to
    all controls in the application, including the ones in the dialog.
    """

    def __init__(self, test_body: typing.Callable[..., typing.Any]) -> None:
        """Initialize new :class:`~TestThread` instance."""
        super().__init__()
        self.__args: typing.Tuple[typing.Any, ...] = ()
        self.__kwargs: typing.Mapping[str, typing.Any] = {}
        self.__exception: typing.Optional[BaseException] = None
        self.__test_body: typing.Final[typing.Callable[..., typing.Any]] = test_body

    def run(self) -> None:
        """Execute the test body."""
        try:
            self.__test_body(*self.__args, **self.__kwargs)
        except BaseException:  # pylint: disable=broad-except
            exception: BaseException
            traceback: types.TracebackType
            _, exception, traceback = sys.exc_info()  # type: ignore
            self.__exception = exception.with_traceback(traceback)

    def __call__(self, *args: typing.Any, **kwargs: typing.Any) -> None:
        """
        Execute a test.

        This will start a new thread, and exit when the thread is finished. In other words - it behaves exactly as
        calling a test directly.
        """
        self.__args = args
        self.__kwargs = kwargs
        self.__exception = None

        wait: qt_utils.Wait = qt_utils.Wait().for_signals(self.finished)
        with wait:
            self.start()

        try:
            if self.__exception:
                raise self.__exception from None
        finally:
            wait.deleteLater()
            self.deleteLater()


def pytest_runtest_call(item: pytest.Function) -> None:
    """Update tests before execution."""
    item.obj = TestThread(item.obj)
