# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for :mod:`anaconda_navigator.api.external_apps.base`."""

from __future__ import annotations

__all__ = ()

import typing

import pytest

from tests.utilities import parameterization_utils

if typing.TYPE_CHECKING:
    import pytest_mock

    from anaconda_navigator.api.external_apps import parsing_utils as t_parsing_utils


@pytest.fixture(scope='function')
def context(mocker: pytest_mock.MockerFixture) -> typing.Iterable['t_parsing_utils.ParsingContext']:
    """Prepare parsing context for application configurations."""
    from anaconda_navigator.api.external_apps import parsing_utils

    yield parsing_utils.ParsingContext(
        process_api=mocker.MagicMock(),
        user_configuration=mocker.MagicMock(),
    )


@parameterization_utils.cases('app_patches')
def test_app_patch(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
) -> None:
    """Check how AppPatch instances are parsed."""
    from anaconda_navigator.api.external_apps import base

    result: typing.Any = base.BaseApp.parse_configuration(context=context, configuration=configuration, app_name='app')
    assert isinstance(result, base.AppPatch)


@parameterization_utils.cases('installable_apps')
def test_installable_app(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
) -> None:
    """Check how InstallableApp instances are parsed."""
    from anaconda_navigator.api.external_apps import base

    result: typing.Any = base.BaseApp.parse_configuration(context=context, configuration=configuration, app_name='app')
    assert isinstance(result, base.BaseInstallableApp)


@parameterization_utils.cases('invalid_apps')
def test_invalid_app(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
) -> None:
    """Check if invalid configurations raise ValidationErrors."""
    from anaconda_navigator.api.external_apps import base
    from anaconda_navigator.api.external_apps import exceptions

    with pytest.raises(exceptions.ValidationError):
        base.BaseApp.parse_configuration(context=context, configuration=configuration, app_name='app')


@parameterization_utils.cases('web_apps')
def test_web_app(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
) -> None:
    """Check how WebApp instances are parsed."""
    from anaconda_navigator.api.external_apps import base

    result: typing.Any = base.BaseApp.parse_configuration(context=context, configuration=configuration, app_name='app')
    assert isinstance(result, base.BaseWebApp)
