# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for :mod:`anaconda_navigator.api.external_apps.validation`."""

from __future__ import annotations

__all__ = ()

import contextlib
import typing

import pytest


@contextlib.contextmanager
def check_validation_error(field: typing.Tuple[typing.Union[int, str], ...] = ()) -> typing.Iterator[None]:
    """Make sure that ValidationError was raised for an expected field."""
    from anaconda_navigator.api.external_apps import exceptions

    with pytest.raises(exceptions.ValidationError):
        try:
            yield
        except exceptions.ValidationError as exception:
            assert exception.field == field
            raise


def test_base() -> None:
    """Check the behavior of the base of all checkers."""
    from anaconda_navigator.api.external_apps import exceptions
    from anaconda_navigator.api.external_apps import validation

    # valid / no field provided
    validation.OfType(str)('test')
    validation.OfType(int)(123)

    # valid / field only from validator
    validation.OfType(str)('test', field=('t', 'u'))
    validation.OfType(int)(123, field=('w', 1, 'x'))

    # valid / field only from context
    with exceptions.ValidationError.with_field(name=('c', 10)):
        validation.OfType(str)('test')
        validation.OfType(int)(123)

    # valid / field from context and validator
    with exceptions.ValidationError.with_field(name=('d', 100)):
        validation.OfType(str)('test', field=('i', -1))
        validation.OfType(int)(123, field=('o', -7))

    # invalid / no field provided
    with check_validation_error():
        validation.OfType(str)(123)
    with check_validation_error():
        validation.OfType(int)('test')

    # invalid / field only from validator
    with check_validation_error(field=('a', 1)):
        validation.OfType(str)(123, field=('a', 1))
    with check_validation_error(field=('b', 2)):
        validation.OfType(int)('test', field=('b', 2))

    # invalid / field only from context
    with check_validation_error(field=('c', 10)):
        with exceptions.ValidationError.with_field(name=('c', 10)):
            validation.OfType(str)(123)
    with check_validation_error(field=('d', 11)):
        with exceptions.ValidationError.with_field(name=('d', 11)):
            validation.OfType(int)('test')

    # invalid / field from context and validator
    with check_validation_error(field=('x', 1, 'y', 2)):
        with exceptions.ValidationError.with_field(name=('x', 1)):
            validation.OfType(str)(123, field=('y', 2))
    with check_validation_error(field=('u', 'v', 100, 'w', 101)):
        with exceptions.ValidationError.with_field(name=('u', 'v')):
            validation.OfType(int)('test', field=(100, 'w', 101))


def test_all_of() -> None:
    """Check the behavior of `AllOf` composite checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.AllOf = (
        validation.OfType(list) &
        validation.OfLength(at_least=1, at_most=2) &
        validation.EachValue(
            validation.OfChoices('abc', 'xyz', 'qwerty') &
            validation.OfLength(3)
        )
    )

    validator(['abc'])
    validator(['xyz'])
    validator(['xyz', 'abc'])

    with check_validation_error():  # invalid type
        validator('test')
    with check_validation_error():  # invalid length
        validator([])
    with check_validation_error(field=(0,)):  # invalid length of an item
        validator(['qwerty'])
    with check_validation_error():  # invalid length
        validator(['xyz', 'xyz', 'xyz'])
    with check_validation_error(field=(0,)):  # invalid item choice
        validator(['hello', 'world'])
    with check_validation_error(field=(0,)):  # invalid item choice
        validator([1, 'xyz'])
    with check_validation_error(field=(1,)):  # invalid item choice
        validator(['abc', 2])


def test_any_of() -> None:
    """Check the behavior of `AnyOf` composite checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.AnyOf = (
        validation.OfLength(at_least=3, at_most=5) |
        validation.OfChoices('x', 'y', 'z') |
        validation.OfType(int)
    )

    validator('abc')
    validator('abcd')
    validator('abcde')
    validator('x')
    validator('y')
    validator('z')
    validator(1)
    validator(11)
    validator(121)

    with check_validation_error():
        validator('qwerty')
    with check_validation_error():
        validator('xx')
    with check_validation_error():
        validator(None)


def test_one_of() -> None:
    """Check the behavior of `OneOf` composite checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.OneOf = (
        validation.OfLength(at_least=2, at_most=4) ^
        validation.OfChoices('x', 'xx', 'xxx', 'xxxx', 'xxxxx', 'xxxxxx') ^
        validation.OfChoices('xxxxxx', 'xXx')
    )

    validator('x')
    validator('xxxxx')
    validator('123')

    with check_validation_error():
        validator('xx')
    with check_validation_error():
        validator('xxx')
    with check_validation_error():
        validator('xXx')
    with check_validation_error():
        validator('xxxx')
    with check_validation_error():
        validator('xxxxxx')


def test_is_empty_mapping() -> None:
    """Check the behavior of `IsEmptyMapping` checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.IsEmptyMapping = validation.IsEmptyMapping()

    validator({})

    with check_validation_error():
        validator({'a': 1})
    with check_validation_error():
        validator({'x': 1, 'y': 2, 'z': 3})


def test_of_choices() -> None:
    """Check the behavior of `OfChoices` checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.OfChoices = validation.OfChoices('a', 'b', 'xyz', 'uvw')

    validator('a')
    validator('b')
    validator('xyz')
    validator('uvw')

    with check_validation_error():
        validator(None)
    with check_validation_error():
        validator(123)
    with check_validation_error():
        validator('abc')
    with check_validation_error():
        validator('rgb')
    with check_validation_error():
        validator('xyzw')


def test_of_length() -> None:
    """Check the behavior of `OfLength` checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.OfLength = validation.OfLength(at_least=2, at_most=5)

    validator('ab')
    validator('abc')
    validator('abcd')
    validator('abcde')

    with check_validation_error():
        validator('a')
    with check_validation_error():
        validator('abcdef')
    with check_validation_error():
        validator('abcdefg')


def test_of_type() -> None:
    """Check the behavior of `OfType` checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.OfType = validation.OfType(str, None)

    validator(None)
    validator('xyz')
    validator('qwerty')

    with check_validation_error():
        validator(123)
    with check_validation_error():
        validator({})


def test_each_pair() -> None:
    """Check the behavior of `EachPair` checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.EachPair = validation.EachPair(
        key=validation.OfType(str) & validation.OfLength(at_least=3, at_most=5),
        value=validation.OfChoices('first', 'second', 'third'),
    )

    validator({})
    validator({'abc': 'first', 'xyz': 'second', 'uvw': 'third'})

    with check_validation_error(field=('rgb',)):
        validator({'abc': 'first', 'xyz': 'second', 'uvw': 'third', 'rgb': 'fourth'})
    with check_validation_error(field=('uvw',)):
        validator({'abc': 'first', 'xyz': 'second', 'uvw': 'custom'})
    with check_validation_error(field=('qwerty',)):
        validator({'abc': 'first', 'xyz': 'second', 'qwerty': 'third'})


def test_each_value() -> None:
    """Check the behavior of `EachValue` checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.EachValue = validation.EachValue(
        value=validation.OfType(str) & validation.OfLength(at_least=2, at_most=3),
    )

    validator([])
    validator(['xyz', 'rgb', 'uvw', 'abc'])

    with check_validation_error(field=(1,)):
        validator(['xyz', 1, 'uvw'])
    with check_validation_error(field=(2,)):
        validator(['abc', 'ab', 'a'])
    with check_validation_error(field=(0,)):
        validator(['hello'])


def test_item_getter() -> None:
    """Check the behavior of `ItemGetter` helper checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.ItemGetter = validation.ItemGetter('key')

    assert validator({'key': 'abcde'}) == 'abcde'
    with check_validation_error():
        validator({})

    validator = validation.ItemGetter('key', default='default')
    assert validator({'key': 'xyzw'}) == 'xyzw'
    assert validator({}) == 'default'

    collection = {'key': 'a'}
    assert validator(collection) == 'a'
    assert collection == {'key': 'a'}

    converter: validation.Converter = validator.convert(validation.OfType(str))
    assert converter({'key': 'rgba'}) == 'rgba'
    assert converter({}) == 'default'
    with check_validation_error(('key',)):
        converter({'key': 123})


def test_item_popper() -> None:
    """Check the behavior of `ItemPopper` helper checker."""
    from anaconda_navigator.api.external_apps import validation

    validator: validation.ItemPopper = validation.ItemPopper('key')

    assert validator({'key': 'qwerty'}) == 'qwerty'
    with check_validation_error():
        validator({})

    validator = validation.ItemPopper('key', default='fallback')
    assert validator({'key': 'xyzw'}) == 'xyzw'
    assert validator({}) == 'fallback'

    collection = {'key': 'a', 'other': 'b'}
    assert validator(collection) == 'a'
    assert collection == {'other': 'b'}
