# -*- coding: utf-8 -*-

"""
Utilities to manage global module imports.

*Primary purpose of such utilities*: is to make sure that each test has its own instance of a module.

*Reason*: some packages have module-level singletons, which may keep state between tests. This should not be the case,
as each test must be independent.

One way to deal with it - is to track each singleton and destroy it before each test that requires it. But keeping track
of all such instances is unreasonably hard task which also requires periodic checks if there is anything added that
should also be tracked.

Other way - is to make sure that each test gets its own copies of modules with only personal data. Thus - each singleton
and component will be unique for each test, eliminating ths possibility to share data between tests. This may require
more time to run all tests, but it should ensure test independence without additional effort.
"""

from __future__ import annotations

__all__ = ['ModuleManager']

import fnmatch
import gc
import typing
import sys


class ModuleManager:
    """
    Manager for a subset of imports.

    Each instance is initialized with a set of patterns, by which modules might be selected. Patterns support Unix
    shell-style wildcards (see `fnmatch documentation <https://docs.python.org/3/library/fnmatch.html>`_ for more
    details).
    """

    __slots__ = ('__patterns',)

    def __init__(self, *patterns: str) -> None:
        """Initialize new :class:`~ImportManager` instance."""
        self.__patterns: typing.Final[typing.Tuple[str, ...]] = patterns

    @property
    def patterns(self) -> typing.Tuple[str, ...]:  # noqa: D401
        """List of module name patterns which should be managed by :class:`~ModuleManager`."""
        return self.__patterns

    def check(self) -> bool:
        """Check if there is at least a single module imported which matches one of :attr:`~ModuleManager.patterns`."""
        try:
            next(self.search())
        except StopIteration:
            return False
        return True

    def release(self) -> typing.List[str]:
        """Release all modules that match at least one of :attr:`~ModuleManager.patterns`."""
        result: typing.List[str] = list(self.search())

        module: str
        for module in result:
            del sys.modules[module]

        gc.collect()

        return result

    def search(self) -> typing.Iterator[str]:
        """Iterate through all modules, which match at least one of :attr:`~ModuleManager.patterns`."""
        module: str
        for module in list(sys.modules):
            if any(fnmatch.fnmatchcase(module, pattern) for pattern in self.__patterns):
                yield module
