# -*- coding: utf-8 -*-

"""Core definitions for predicates."""

from __future__ import annotations

__all__ = ['GenericPredicate']

import abc
import typing

T_contra = typing.TypeVar('T_contra', contravariant=True)


class Predicate(typing.Protocol[T_contra]):  # pylint: disable=too-few-public-methods
    """Common interface for predicates."""

    def __call__(self, instance: T_contra) -> bool:
        """Check if `instance` meets a condition."""


CompositeT = typing.TypeVar('CompositeT', bound='CompositePredicate')


def flatten(
        output: typing.Type['CompositePredicate[T_contra]'],
        *args: 'Predicate[T_contra]',
) -> 'CompositePredicate[T_contra]':
    """
    Remove unnecessary nesting in :class:`~CompositePredicate`.

    This function creates new instance of `output` with `args` as a children. If any of `args` is an instance of
    `output` - its children would be used instead of the arg itself.

    E.g. attempting to create :code:`And(And(A, B), C, And(D, E))` would result in creating :code:`And(A, B, C, D, E)`.
    """
    result: typing.List['Predicate[T_contra]'] = []

    arg: 'Predicate[T_contra]'
    for arg in args:
        if type(arg) == output:  # noqa E721  # pylint: disable=unidiomatic-typecheck
            result.extend(arg.children)
        else:
            result.append(arg)

    return output(*result)


class GenericPredicate(typing.Generic[T_contra], metaclass=abc.ABCMeta):
    """
    Common base for class-based predicates.

    This base also supports binary operators like :code:`A & B`, :code:`A | B` and :code:`~A`.
    """

    __slots__ = ()

    @abc.abstractmethod
    def __call__(self, instance: T_contra) -> bool:
        """Check if `instance` meets a condition."""

    def __and__(self, other: 'Predicate[T_contra]') -> 'Predicate[T_contra]':
        """Require this and `other` predicates to return :code:`True`."""
        return flatten(And, self, other)

    def __neg__(self) -> 'Predicate[T_contra]':
        """Create a predicate returning opposite value."""
        if type(self) == Not:  # noqa E721  # pylint: disable=unidiomatic-typecheck
            return self.child  # pylint: disable=no-member
        return Not(self)

    def __or__(self, other: 'Predicate[T_contra]') -> 'Predicate[T_contra]':
        """Require at least one predicate to return :code:`True`."""
        return flatten(Or, self, other)

    def __rand__(self, other: 'Predicate[T_contra]') -> 'Predicate[T_contra]':
        """Require this and `other` predicates to return :code:`True`."""
        return flatten(And, other, self)

    def __ror__(self, other: 'Predicate[T_contra]') -> 'Predicate[T_contra]':
        """Require at least one predicate to return :code:`True`."""
        return flatten(Or, other, self)


class CompositePredicate(  # pylint: disable=too-few-public-methods
    GenericPredicate[T_contra],
    typing.Generic[T_contra],
    metaclass=abc.ABCMeta,
):
    """Base for predicates that combine values of multiple children predicates."""

    __slots__ = ('__children',)

    def __init__(self, *args: Predicate[T_contra]) -> None:
        """Initialize new :class:`~CompositePredicate` instance."""
        self.__children: typing.Final[typing.Tuple[Predicate[T_contra], ...]] = args

    @property
    def children(self) -> typing.Tuple['Predicate[T_contra]', ...]:  # noqa: D401
        """Child predicates, which results should be combined."""
        return self.__children


class ProxyPredicate(  # pylint: disable=too-few-public-methods
    GenericPredicate[T_contra],
    typing.Generic[T_contra],
    metaclass=abc.ABCMeta,
):
    """Base for predicates that modify value of another predicate."""

    __slots__ = ('__child',)

    def __init__(self, child: Predicate[T_contra]) -> None:
        """Initialize new :class:`~ProxyPredicate` instance."""
        self.__child: typing.Final[Predicate[T_contra]] = child

    @property
    def child(self) -> 'Predicate[T_contra]':  # noqa: D401
        """Child predicate, which value should be modified."""
        return self.__child


class And(CompositePredicate[T_contra], typing.Generic[T_contra]):
    """Combined predicate which requires all child predicates to return :code:`True`."""

    __slots__ = ()

    def __call__(self, instance: T_contra) -> bool:
        """Check if `instance` meets a condition."""
        return all(child(instance) for child in self.children)


class Or(CompositePredicate[T_contra], typing.Generic[T_contra]):
    """Combined predicate which requires at least one child predicate to return :code:`True`."""

    __slots__ = ()

    def __call__(self, instance: T_contra) -> bool:
        """Check if `instance` meets a condition."""
        return any(child(instance) for child in self.children)


class Not(ProxyPredicate[T_contra], typing.Generic[T_contra]):
    """Proxy predicate which inverts the result of a child predicate."""

    __slots__ = ()

    def __call__(self, instance: T_contra) -> bool:
        """Check if `instance` meets a condition."""
        return not self.child(instance)
