# -*- coding: utf-8 -*-

"""Predicates for checking text content of Qt widgets."""

from __future__ import annotations

__all__ = ['MatchTarget', 'ContainsText', 'MatchesText', 'WithText']

import abc
import enum
import re
import typing

from . import core

if typing.TYPE_CHECKING:
    from .. import qwidget_collection


class TextPredicate(core.GenericPredicate['qwidget_collection.Item'], metaclass=abc.ABCMeta):
    """
    Common base for all text content checking predicates.

    :param case_sensitive: Preserve case while checking text content.
    """

    __slots__ = ('__case_sensitive',)

    def __init__(self, *, case_sensitive: bool = False) -> None:
        """Initialize new :class:`~TextPredicate` instance."""
        self.__case_sensitive: typing.Final[bool] = case_sensitive

    @property
    def case_sensitive(self) -> bool:  # noqa: D401
        """Preference for case-sensitive check."""
        return self.__case_sensitive

    @abc.abstractmethod
    def _validate(self, value: str) -> bool:
        """Perform validation of a text value."""

    def __call__(self, instance: 'qwidget_collection.Item') -> bool:
        """Check if `instance` meets a condition."""
        current: typing.Optional[str]
        try:
            current = instance.text()
        except (AttributeError, TypeError):
            return False
        if current is None:
            return False
        if not self.__case_sensitive:
            current = current.lower()
        return self._validate(current)


class StringPredicate(TextPredicate, metaclass=abc.ABCMeta):
    """
    Base for predicates that check text content against a string value.

    :param value: String to check text content against.
    :param case_sensitive: Preserve case while checking text content.

    .. note::

        Internal check might be case-sensitive as all required case changes are already done by the base itself.
    """

    __slots__ = ('__value',)

    def __init__(self, value: str, *, case_sensitive: bool = False) -> None:
        """Initialize new :class:`~StringPredicate` instance."""
        super().__init__(case_sensitive=case_sensitive)

        if not case_sensitive:
            value = value.lower()

        self.__value: typing.Final[str] = value

    @property
    def value(self) -> str:  # noqa: D401
        """String value to check content against."""
        return self.__value


class RegexPredicate(TextPredicate, metaclass=abc.ABCMeta):
    """
    Base for predicates that check text content against a regular expression.

    :param pattern: Regular expression to use as a pattern for text checks.
    :param case_sensitive: Preserve case while checking text content.
    :param multiline: Pattern should be used to check each separate line in the text content instead of the whole
                      text content.
    """

    __slots__ = ('__pattern',)

    def __init__(
            self,
            pattern: typing.Union[str, typing.Pattern[str]],
            *,
            case_sensitive: bool = False,
            multiline: bool = False,
    ) -> None:
        """Initialize new :class:`~RegexPredicate` instance."""
        super().__init__(case_sensitive=case_sensitive)

        if not isinstance(pattern, str):
            pattern = pattern.pattern

        flags: re.RegexFlag = re.RegexFlag(0)
        if not case_sensitive:
            flags |= re.RegexFlag.IGNORECASE
        if multiline:
            flags |= re.RegexFlag.MULTILINE

        self.__pattern: typing.Final[typing.Pattern[str]] = re.compile(pattern, flags)

    @property
    def pattern(self) -> typing.Pattern[str]:  # noqa: D401
        """Prepared and compiled regular expression to check text content with."""
        return self.__pattern


class MatchTarget(enum.IntEnum):
    """
    Which part of the text content should be checked with regular expression.

    .. attribute:: ANYWHERE

        Any substring of a text content must match a pattern.

    .. attribute:: FULL

        Complete text content value must match a pattern.

    .. attribute:: START

        Substring that starts from the beginning of a text content must match a pattern.
    """

    ANYWHERE = enum.auto()
    FULL = enum.auto()
    START = enum.auto()


class ContainsText(StringPredicate):
    """
    Check if text content contains a substring.

    :param value: String to check text content against.
    :param case_sensitive: Preserve case while checking text content.
    """

    __slots__ = ()

    def _validate(self, value: str) -> bool:
        """Perform validation of a text value."""
        return self.value in value


class MatchesText(RegexPredicate):
    """
    Check text content of a Qt widget with a regular expression.

    :param pattern: Regular expression to use as a pattern for text checks.
    :param case_sensitive: Preserve case while checking text content.
    :param multiline: Pattern should be used to check each separate line in the text content instead of the whole
                      text content.
    :param target: Which part of text content must be checked with a regular expression. See :class:`~MatchTarget` for
                   available options.
    """

    __slots__ = ('__target',)

    def __init__(
            self,
            pattern: typing.Union[str, typing.Pattern[str]],
            *,
            case_sensitive: bool = False,
            multiline: bool = False,
            target: MatchTarget = MatchTarget.FULL,
    ) -> None:
        """Initialize new :class:`~MatchesText` instance."""
        super().__init__(pattern, case_sensitive=case_sensitive, multiline=multiline)

        self.__target: typing.Final[MatchTarget] = target

    @property
    def target(self) -> MatchTarget:  # noqa: D401
        """Which part of text content is checked by this predicate."""
        return self.__target

    def _validate(self, value: str) -> bool:
        """Perform validation of a text value."""
        if self.target == MatchTarget.FULL:
            return bool(self.pattern.fullmatch(value))
        if self.target == MatchTarget.ANYWHERE:
            return bool(self.pattern.search(value))
        if self.target == MatchTarget.START:
            return bool(self.pattern.match(value))
        raise ValueError()


class WithText(StringPredicate):
    """
    Check if complete text content equals to expected value.

    :param value: String to check text content against.
    :param case_sensitive: Preserve case while checking text content.
    """

    __slots__ = ()

    def _validate(self, value: str) -> bool:
        """Perform validation of a text value."""
        return self.value == value
