# -*- coding: utf-8 -*-

"""Utilities to wait for some events on a testing thread."""

from __future__ import annotations

__all__ = ['Wait']

import abc
import itertools
import time
import types
import typing

import attr
from qtpy import QtCore

ContextT_co = typing.TypeVar('ContextT_co', bound='EventContext', covariant=True)
ContextT_contra = typing.TypeVar('ContextT_contra', bound='EventContext', contravariant=True)
ResultT = typing.TypeVar('ResultT')
ResultT_co = typing.TypeVar('ResultT_co', covariant=True)


class EventConstructor(typing.Protocol[ContextT_contra, ResultT_co]):  # pylint: disable=too-few-public-methods
    """Common interface for constructors of :class:`~EventListener` instance."""

    def __call__(self, context: ContextT_contra) -> EventListener[ContextT_contra, ResultT_co]:
        """Construct :class:`~EventListener` instance."""


# Abstract base for different events

@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class EventContext:  # pylint: disable=too-few-public-methods
    """Common base for contexts used by :class:`~Event` and :class:`~EventListener` instances."""

    identity: str = attr.ib(default='')


class EventListener(QtCore.QObject, typing.Generic[ContextT_co, ResultT_co]):
    """
    Common base for event listeners.

    Primary function of such listeners - is to emit the :attr:`~EventListener.sig_done` signal as soon as event, for
    which this instance awaits, happens.

    Lifecycle of each listener is pretty simple:

    - it starts listening with a :meth:`~EventListener.start` call.
    - it captures event and registers corresponding output by calling a :meth:`~EventListener._done` function.
    - it stops listening with a :meth:`~EventListener.stop` call.

    After listening is stopped - it is not expected that this instance would be reused. Instead - new instance would be
    created with :meth:`~Event.prepare`.
    """

    sig_done = QtCore.Signal(QtCore.QObject)

    def __init__(self, context: ContextT_co) -> None:
        """Initialize new :class:`~EventListener` instance."""
        super().__init__()
        self.__context: typing.Final[ContextT_co] = context
        self.__done: bool = False
        self.__exception: typing.Optional[BaseException] = None
        self.__result: typing.Optional[ResultT_co] = None

    @property
    def context(self) -> ContextT_co:  # noqa: D401
        """Event listening context."""
        return self.__context

    @property
    def done(self) -> bool:  # noqa: D401
        """Current status of the event listening - still waiting for an event, or already done waiting."""
        return self.__done

    @property
    def exception(self) -> typing.Optional[BaseException]:  # noqa: D401
        """Exception raised during event listening."""
        return self.__exception

    @property
    def result(self) -> typing.Any:  # noqa: D401
        """Result of event listening."""
        return self.__result

    def start(self) -> None:
        """Start listening to an event."""

    def stop(self) -> None:
        """Stop listening to an event."""

    def _done(
            self,
            result: typing.Optional[ResultT_co] = None,
            *,
            exception: typing.Optional[BaseException] = None,
    ) -> None:
        """
        Register event listening output.

        .. warning::

            This must be called only when event happened!

        :param result: Value that should be returned to the user related to this event.
        :param exception: Exception that must be raised because of the event (e.g. :code:`TimeoutError`).
        """
        self.__done = True
        self.__result = result
        self.__exception = exception
        self.sig_done.emit(self)


class Event(typing.Generic[ContextT_co, ResultT_co], metaclass=abc.ABCMeta):
    """
    Common base for events that user may wait for.

    Primary function - is to create new instance of :class:`~EventListener` when user requests waiting for an event.
    """

    __slots__ = ('__context',)

    def __init__(self, context: ContextT_co) -> None:
        """Initialize new :class:`~Event` instance."""
        self.__context: typing.Final[ContextT_co] = context

    @property
    def context(self) -> ContextT_co:  # noqa: D401
        """Event listening context."""
        return self.__context

    @abc.abstractmethod
    def prepare(self) -> EventListener[ContextT_co, ResultT_co]:
        """Create new event listening instance."""


class SimpleEvent(Event[ContextT_co, ResultT_co], typing.Generic[ContextT_co, ResultT_co]):
    """Helper :class:`~Event` which calls an external `constructor` to initialize new :class:`~EventListener`."""

    __slots__ = ('__constructor',)

    def __init__(self, constructor: 'EventConstructor[ContextT_co, ResultT_co]', context: ContextT_co) -> None:
        """Initialize new :class:`~SimpleEvent` instance."""
        super().__init__(context=context)
        self.__constructor: typing.Final[EventConstructor[ContextT_co, ResultT_co]] = constructor

    def prepare(self) -> EventListener[ContextT_co, ResultT_co]:
        """Create new event listening instance."""
        return self.__constructor(context=self.context)


# Negation event

@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class NegationEventContext(EventContext):  # pylint: disable=too-few-public-methods
    """Context for :class:`~NegationEvent`."""

    event: Event[typing.Any, typing.Any]

    identity: str = attr.ib(default='')


class NegationEventListener(EventListener[NegationEventContext, None]):
    """Listener for a :class:`~NegationEvent`."""

    def __init__(self, context: NegationEventContext) -> None:
        """Initialize new :class:`~NegationEventListener` instance."""
        super().__init__(context=context)

        self.__listener: typing.Optional[EventListener[typing.Any, typing.Any]] = None

    def start(self) -> None:
        """Start listening to an event."""
        self.__listener = self.context.event.prepare()
        self.__listener.sig_done.connect(self._check)
        self.__listener.start()

    def stop(self) -> None:
        """Stop listening to an event."""
        if self.__listener is None:
            return

        self.__listener.sig_done.disconnect(self._check)
        self.__listener.stop()
        self.__listener.deleteLater()
        self.__listener = None

    def _check(self, event_listener: EventListener) -> None:  # pylint: disable=unused-argument
        """Check for an event state update."""
        self._done(exception=AssertionError())


class NegationEvent(SimpleEvent[NegationEventContext, None]):
    """
    Make sure that some event doesn't happen.

    Let's say, you have a component that may emit one of multiple signals. You want to check for a particular one, while
    any other one would be considered an error. This event may happen with such cases - if you create a
    :class:`NegationEvent` around unwanted signals - it would raise an error, and you'll get this result immediately
    without any need to wait for a positive signal to timeout.

    :param event: event that should be negated.
    :param identity: unique identifier to use for this event.
    """

    __slots__ = ()

    def __init__(self, context: NegationEventContext) -> None:
        """Initialize new :class:`~NegationEvent` instance."""
        super().__init__(constructor=NegationEventListener, context=context)


# Composite events

@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class CompositeEventContext(EventContext, typing.Generic[ResultT_co]):  # pylint: disable=too-few-public-methods
    """Context for :class:`~CompositeEvent`."""

    events: typing.Sequence[Event[typing.Any, ResultT_co]]
    count: int

    identity: str = attr.ib(default='')


class CompositeEventListener(
    EventListener[CompositeEventContext[ResultT_co], typing.List[typing.Tuple[str, ResultT_co]]],
    typing.Generic[ResultT_co],
):
    """Listener for a :class:`~CompositeEvent`."""

    def __init__(self, context: CompositeEventContext[ResultT_co]) -> None:
        """Initialize new :class:`~CompositeEventListener` instance."""
        super().__init__(context=context)

        self.__count: int = 0
        self.__listeners: typing.List[EventListener[typing.Any, ResultT_co]] = []
        self.__result: typing.List[typing.Tuple[str, ResultT_co]] = []

    def start(self) -> None:
        """Start listening to an event."""
        if self.context.count <= 0:
            self._done(result=[])
            return

        self.__count = self.context.count

        event: Event[typing.Any, ResultT_co]
        for event in self.context.events:
            listener: EventListener[typing.Any, ResultT_co] = event.prepare()
            listener.sig_done.connect(self._check)
            self.__listeners.append(listener)
            listener.start()

    def stop(self) -> None:
        """Stop listening to an event."""
        while self.__listeners:
            listener: EventListener[typing.Any, ResultT_co] = self.__listeners.pop()
            listener.sig_done.disconnect()
            listener.stop()
            listener.deleteLater()

    def _check(self, event_listener: EventListener[typing.Any, ResultT_co]) -> None:
        """Check for an event state update."""
        if event_listener.exception is not None:
            self.__count = 0
            self._done(exception=event_listener.exception)
            return

        self.__result.append((event_listener.context.identity, event_listener.result))

        self.__count -= 1
        if self.__count == 0:
            self._done(result=self.__result)


class CompositeEvent(
    SimpleEvent[CompositeEventContext[ResultT_co], typing.List[typing.Tuple[str, ResultT_co]]],
    typing.Generic[ResultT_co],
):
    """
    Events that awaits for multiple other events to occur.

    :param events: events to wait for.
    :param count: how many child events must occur.
    :param identity: unique identifier to use for this event.
    """

    __slots__ = ()

    def __init__(self, context: CompositeEventContext[ResultT_co]) -> None:
        """Initialize new :class:`~CompositeEvent` instance."""
        super().__init__(constructor=CompositeEventListener, context=context)


# Timeout events

@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class TimeoutEventContext(EventContext):  # pylint: disable=too-few-public-methods
    """Context for :class:`~TimeoutEvent`."""

    interval: int  # msec
    raise_exception: bool = attr.ib(default=False)

    identity: str = attr.ib(default='')


class TimeoutEventListener(EventListener[TimeoutEventContext, None]):
    """Listener for a :class:`~TimeoutEvent`."""

    def __init__(self, context: TimeoutEventContext) -> None:
        """Initialize new :class:`~TimeoutEventListener` instance."""
        super().__init__(context=context)

        self.__timer: typing.Final[QtCore.QTimer] = QtCore.QTimer(self)
        self.__timer.setInterval(self.context.interval)
        self.__timer.setSingleShot(True)
        self.__timer.timeout.connect(self._check)

    def start(self) -> None:
        """Start listening to an event."""
        self.__timer.start()

    def stop(self) -> None:
        """Stop listening to an event."""
        self.__timer.stop()

    def _check(self) -> None:
        """Check for an event state update."""
        if self.context.raise_exception:
            self._done(exception=TimeoutError())
        else:
            self._done()


class TimeoutEvent(SimpleEvent[TimeoutEventContext, None]):
    """
    Wait for a timeout.

    Useful to break waiting for other events if it takes to long.

    :param interval: how many msec wait to raise a timeout.
    :param raise_exception: raise :class:`~TimeoutError` or timeout silently.
    :param identity: unique identifier to use for this event.
    """

    __slots__ = ()

    def __init__(self, context: TimeoutEventContext) -> None:
        """Initialize new :class:`~TimeoutEvent` instance."""
        super().__init__(constructor=TimeoutEventListener, context=context)


# Signal events

@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class SignalEventContext(EventContext):  # pylint: disable=too-few-public-methods
    """Context for :class:`~SignalEvent`."""

    signals: typing.Sequence[QtCore.Signal]
    count: int

    identity: str = attr.ib(default='')


class SignalEventListener(EventListener[SignalEventContext, None]):
    """Listener for a :class:`~SignalEvent`."""

    def __init__(self, context: SignalEventContext) -> None:
        """Initialize new :class:`~SignalEventListener` instance."""
        super().__init__(context=context)

        self.__count: int = 0

    def start(self) -> None:
        """Start listening to an event."""
        if self.context.count <= 0:
            self._done()
            return

        self.__count = self.context.count

        signal: QtCore.Signal
        for signal in self.context.signals:
            signal.connect(self._check)

    def stop(self) -> None:
        """Stop listening to an event."""
        signal: QtCore.Signal
        for signal in self.context.signals:
            signal.disconnect(self._check)

    def _check(self) -> None:
        """Check for an event state update."""
        self.__count -= 1
        if self.__count == 0:
            self._done()


class SignalEvent(SimpleEvent[SignalEventContext, None]):
    """
    Wait for signals to be emitted.

    :param signals: signals to wait for.
    :param count: how many signals must be emitted.
    :param identity: unique identifier to use for this event.
    """

    __slots__ = ()

    def __init__(self, context: SignalEventContext) -> None:
        """Initialize new :class:`~SignalEvent` instance."""
        super().__init__(constructor=SignalEventListener, context=context)


# Condition events

def msec() -> int:
    """Get current value of milliseconds after the epoch."""
    return int(time.time() * 1000)


@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class ConditionEventContext(EventContext, typing.Generic[ResultT_co]):  # pylint: disable=too-few-public-methods
    """Context for :class:`~ConditionEvent`."""

    condition: typing.Callable[[], typing.Optional[ResultT_co]]

    timeout: int = attr.ib(default=10_000)
    wait_before: int = attr.ib(default=0)
    wait_after: int = attr.ib(default=0)
    poll_interval: int = attr.ib(default=200)

    raise_exception: bool = attr.ib(default=False)

    identity: str = attr.ib(default='')


class ConditionEventListener(EventListener[ConditionEventContext[ResultT_co], ResultT_co], typing.Generic[ResultT_co]):
    """Listener for a :class:`~ConditionEvent`."""

    def __init__(self, context: ConditionEventContext[ResultT_co]) -> None:
        """Initialize new :class:`~ConditionEventListener` instance."""
        super().__init__(context=context)

        self.__timer: QtCore.QTimer = QtCore.QTimer()
        self.__timer.setSingleShot(True)
        self.__timer.timeout.connect(self._check)

        self.__timeout: int = -1
        self.__return: int = -1

    def start(self) -> None:
        """Start listening to an event."""
        interval: int = self.context.wait_before + self.context.timeout

        self.__timeout = msec() + interval

        if self.context.wait_before > 0:
            self.__timer.setInterval(self.context.wait_before)
            self.__timer.start()
            return

        self._check()

    def stop(self) -> None:
        """Stop listening to an event."""
        self.__timer.stop()

    def _check(self) -> None:
        """Check for an event state update."""
        result: typing.Optional[ResultT_co] = self.context.condition()

        now: int = msec()
        if self.__return >= 0:
            if result is None:
                self.__return = -1
            elif now >= self.__return:
                self._done(result=result)
                return
            else:
                self.__timer.setInterval(min(self.context.poll_interval, self.__return - now))
                self.__timer.start()
                return

        if result is not None:
            if self.context.wait_after > 0:
                self.__return = now + self.context.wait_after
                self.__timer.setInterval(min(self.context.poll_interval, self.context.wait_after))
                self.__timer.start()
                return

            self._done(result=result)
            return

        if now < self.__timeout:
            self.__timer.setInterval(min(self.context.poll_interval, self.__timeout - now))
            self.__timer.start()
            return

        if self.context.raise_exception:
            self._done(exception=TimeoutError())
            return

        self._done()


class ConditionEvent(SimpleEvent[ConditionEventContext[ResultT_co], ResultT_co], typing.Generic[ResultT_co]):
    """
    Wait for some condition.

    This would poll condition until it returns a not-:code:`None` value.

    :param condition: condition function, that should return awaited instance. If condition is not met - :code:`None`
                      must be returned.
    :param wait_before: additional pause (in msec) before checking a condition for the first time.
    :param wait_after: additional pause (in msec) after condition is met. It expects a condition to still return a
                       result during all this waiting period. If condition breaks during this waiting period (returns
                       :code:`None`) - this would restore the waiting for the condition.
    :param poll_interval: interval between condition checks.
    :param raise_exception: raise a :class:`~TimeoutError` after waiting intervals. Otherwise - won't raise even an
                            event.
    :param identity: unique identifier to use for this event.
    """

    __slots__ = ()

    def __init__(self, context: ConditionEventContext[ResultT_co]) -> None:
        """Initialize new :class:`~ConditionEvent` instance."""
        super().__init__(constructor=ConditionEventListener, context=context)


# Common interface for all stuff

WaitT = typing.TypeVar('WaitT', bound='Wait')


class Wait(QtCore.QObject, typing.Generic[ResultT_co]):
    """
    Waiter for external events.

    Create new instance, chain what to wait for with :code:`wait_for_*` or :code:`except_for` methods, and then
    call :meth:`~Wait.exec_` to start waiting.

    Instead of calling :meth:`~Wait.exec_` directly, it is also possible to use it as a context manager. This would
    start waiting process right after the context body.
    """

    __slots__ = ('__events', '__listener')

    def __init__(
            self,
            *args: typing.Union[Event[typing.Any, ResultT_co], typing.Iterable[Event[typing.Any, ResultT_co]]],
    ) -> None:
        """Initialize new :class:`~Wait` instance."""
        super().__init__()
        self.__events: typing.Tuple[Event[typing.Any, ResultT_co], ...] = tuple(flatten(args))
        self.__listener: typing.Optional[CompositeEventListener[ResultT_co]] = None

    def except_for(
            self,
            event: typing.Union[Event[typing.Any, typing.Any], typing.Iterable[Event[typing.Any, typing.Any]]],
            *,
            identity: str = '',
    ) -> 'Wait[typing.Optional[ResultT_co]]':
        """
        Chain a :class:`~NegationEvent`.

        :param event: event that should be negated.
        :param identity: unique identifier to use for this event.
        """
        events: typing.Tuple[Event, ...] = tuple(flatten(event))

        if len(events) == 1:
            event = events[0]
        else:
            inner_context: CompositeEventContext = CompositeEventContext(events=events, count=1)
            event = CompositeEvent(context=inner_context)

        context: NegationEventContext = NegationEventContext(event=event, identity=identity)
        event = NegationEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def exec_(self) -> typing.Tuple[str, ResultT_co]:
        """
        Start waiting process.

        Exits as soon as first event occurs. If any error is raised by the event - this function would raise it as well.
        Otherwise - it would return an identity of an event and value returned by its listener.
        """
        self.__start()
        self.__wait()
        try:
            if self.__listener is None:
                raise ValueError('listener closed too soon')
            if self.__listener.exception is not None:
                raise self.__listener.exception from None
            return self.__listener.result[0]
        finally:
            self.__stop()

    def for_all_of(
            self,
            *args: typing.Union[Event[typing.Any, ResultT], typing.Iterable[Event[typing.Any, ResultT]]],
            identity: str = '',
    ) -> 'Wait[typing.Union[ResultT_co, typing.List[typing.Tuple[str, ResultT]]]]':
        """
        Chain a :class:`~CompositeEvent` that waits for at all of provided events to occur.

        :param args: events to wait for.
        :param identity: unique identifier to use for this event.
        """
        events: typing.Tuple[Event, ...] = tuple(flatten(args))

        context: CompositeEventContext[ResultT] = CompositeEventContext(
            events=events,
            count=len(events),
            identity=identity,
        )
        event: CompositeEvent[ResultT] = CompositeEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def for_condition(  # pylint: disable=too-many-arguments
            self,
            condition: typing.Callable[[], typing.Optional[ResultT]],
            *,
            timeout: int = 10_000,
            wait_before: int = 0,
            wait_after: int = 0,
            poll_interval: int = 200,
            raise_exception: bool = False,
            identity: str = '',
    ) -> 'Wait[typing.Union[ResultT_co, ResultT]]':
        """
        Chain a :class:`~ConditionEvent` that waits for a condition to meet.

        :param condition: condition function, that should return awaited instance. If condition is not met -
                          :code:`None` must be returned.
        :param wait_before: additional pause (in msec) before checking a condition for the first time.
        :param wait_after: additional pause (in msec) after condition is met. It expects a condition to still return a
                           result during all this waiting period. If condition breaks during this waiting period
                           (returns :code:`None`) - this would restore the waiting for the condition.
        :param poll_interval: interval between condition checks.
        :param raise_exception: raise a :class:`~TimeoutError` after waiting intervals. Otherwise - won't raise even an
                                event.
        :param identity: unique identifier to use for this event.
        """
        context: ConditionEventContext[ResultT] = ConditionEventContext(
            condition=condition,
            timeout=timeout,
            wait_before=wait_before,
            wait_after=wait_after,
            poll_interval=poll_interval,
            raise_exception=raise_exception,
            identity=identity,
        )
        event: ConditionEvent[ResultT] = ConditionEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def for_one_of(
            self,
            *args: typing.Union[Event[typing.Any, ResultT], typing.Iterable[Event[typing.Any, ResultT]]],
            identity: str = '',
    ) -> 'Wait[typing.Union[ResultT_co, typing.List[typing.Tuple[str, ResultT]]]]':
        """
        Chain a :class:`~CompositeEvent` that waits for at least one of provided events to occur.

        :param args: events to wait for.
        :param identity: unique identifier to use for this event.
        """
        events: typing.Tuple[Event[typing.Any, ResultT], ...] = tuple(flatten(args))

        if not events:
            raise ValueError('at least one event must be provided')

        context: CompositeEventContext[ResultT] = CompositeEventContext(events=events, count=1, identity=identity)
        event: CompositeEvent[ResultT] = CompositeEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def for_signals(
            self,
            *args: QtCore.Signal,
            count: typing.Optional[int] = None,
            identity: str = '',
    ) -> 'Wait[typing.Optional[ResultT_co]]':
        """
        Chain a :class:`~SignalEvent` that waits for multiple signals to be emitted.

        :param args: signals to wait for.
        :param count: how many signals must be emitted.
        :param identity: unique identifier to use for this event.
        """
        if count is None:
            count = len(args)

        context: SignalEventContext = SignalEventContext(signals=args, count=count, identity=identity)
        event: SignalEvent = SignalEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def for_some_of(
            self,
            *args: typing.Union[Event[typing.Any, ResultT], typing.Iterable[Event[typing.Any, ResultT]]],
            count: int,
            identity: str = '',
    ) -> 'Wait[typing.Union[ResultT_co, typing.List[typing.Tuple[str, ResultT]]]]':
        """
        Chain a :class:`~CompositeEvent` that waits for multiple events to occur.

        :param args: events to wait for.
        :param count: how many child events must occur.
        :param identity: unique identifier to use for this event.
        """
        if count < 0:
            raise ValueError(f'count must be at least 0, not {count}')

        events: typing.Tuple[Event[typing.Any, ResultT], ...] = tuple(flatten(args))

        if count > len(events):
            raise ValueError(f'count must be at most {len(events)}, not {count}')

        context: CompositeEventContext[ResultT] = CompositeEventContext(events=events, count=count, identity=identity)
        event: CompositeEvent[ResultT] = CompositeEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def for_timeout(
            self,
            interval: int,
            *,
            raise_exception: bool = False,
            identity: str = '',
    ) -> 'Wait[typing.Optional[ResultT_co]]':
        """
        Chain a :class:`~TimeoutEvent` that is used to break waiting after some period of time.

        :param interval: how many msec wait to raise a timeout.
        :param raise_exception: raise :class:`~TimeoutError` or timeout silently.
        :param identity: unique identifier to use for this event.
        """
        context: TimeoutEventContext = TimeoutEventContext(
            interval=interval,
            raise_exception=raise_exception,
            identity=identity,
        )
        event: TimeoutEvent = TimeoutEvent(context=context)
        return Wait(itertools.chain(self.__events, [event]))

    def __start(self) -> None:
        """Start listening for events."""
        if self.__listener is not None:
            raise TypeError('listener already started')

        context: CompositeEventContext[ResultT_co] = CompositeEventContext(events=self.__events, count=1)
        self.__listener = CompositeEventListener(context=context)
        self.__listener.start()

    def __wait(self) -> None:
        """Wait until event."""
        if self.__listener is None:
            raise TypeError('listener closed too soon')

        loop: QtCore.QEventLoop = QtCore.QEventLoop()
        self.__listener.sig_done.connect(loop.quit)
        if not self.__listener.done:
            loop.exec_()
        self.__listener.sig_done.disconnect()
        loop.deleteLater()

    def __stop(self) -> None:
        """Stop listening for events and cleanup."""
        if self.__listener is None:
            raise TypeError('listener closed too soon')

        self.__listener.stop()
        self.__listener.deleteLater()
        self.__listener = None

    def __enter__(self: WaitT) -> WaitT:
        """Enter a waiting context."""
        self.__start()
        return self

    def __exit__(
            self,
            exc_type: typing.Optional[typing.Type[BaseException]] = None,
            exc_val: typing.Optional[BaseException] = None,
            exc_tb: typing.Optional[types.TracebackType] = None,
    ) -> None:
        """
        Exit from waiting context.

        Launches waiting process if no errors happened during the context body.
        """
        if exc_val is None:
            self.__wait()
        self.__stop()

    @typing.overload
    def __getitem__(self, index: int) -> Event:
        """Retrieve single event from an instance."""

    @typing.overload
    def __getitem__(self, index: slice) -> 'Wait[ResultT_co]':
        """Retrieve a subsequence of events."""

    def __getitem__(self, index: typing.Union[int, slice]) -> typing.Union[Event, 'Wait[ResultT_co]']:
        """Retrieve a content."""
        if isinstance(index, slice):
            return Wait(self.__events[index])
        return self.__events[index]

    def __iter__(self) -> typing.Iterator[Event]:
        """Iterate through a stored events."""
        return iter(self.__events)

    def __len__(self) -> int:
        """Retrieve total number of events stored in an instance."""
        return len(self.__events)


def flatten(
        *args: typing.Union[Event[typing.Any, ResultT], typing.Iterable[typing.Any]],
) -> typing.Iterator[Event[typing.Any, ResultT]]:
    """Iterate over nested collections of events, returning only :class:`~Event` instances."""
    queue: typing.List[typing.Union[Event[typing.Any, ResultT], typing.Iterable[typing.Any]]] = list(reversed(args))
    queue.reverse()

    while queue:
        arg: typing.Union[Event[typing.Any, ResultT], typing.Iterable[typing.Any]] = queue.pop()

        if isinstance(arg, Event):
            yield arg
            continue

        queue.extend(reversed(list(arg)))
