# -*- coding: utf-8 -*-

"""Utilities to create a temporary home directory."""

from __future__ import annotations

__all__ = ['Manager', 'DirectoryManager', 'CompositeManager']

import abc
import itertools
import os
import shutil
import tarfile
import tempfile
import typing
import uuid


T = typing.TypeVar('T')


SNAPSHOT_EXTENSION: typing.Final[str] = '.tar'


def split_path(path: str) -> typing.Sequence[str]:
    """Split a file path into sequence of directory/file names."""
    left: str
    right: str
    result: typing.List[str] = []
    left, right = os.path.split(path)
    while right:
        result.append(right)
        left, right = os.path.split(left)
    if left:
        result.append(left)
    result.reverse()
    return result


def common_length(*args: typing.Sequence[T]) -> int:
    """Find length of the common prefix of multiple sequences."""
    return sum(1 for _ in itertools.takewhile(lambda item: len(set(item)) == 1, zip(*args)))


def remove_prefixes(source: typing.Mapping[str, typing.Sequence[str]]) -> typing.Mapping[str, typing.Sequence[str]]:
    """Remove common prefixes from values of mapping."""
    offset: int = common_length(*source.values())
    return {
        key: value[offset:]
        for key, value in source.items()
    }


class Manager(metaclass=abc.ABCMeta):
    """General interface for directory managers."""

    __slots__ = ()

    @abc.abstractmethod
    def activate(self, environment: str) -> None:
        """
        Activate existing environment.

        :param environment: Name of the environment to activate.
        """

    @abc.abstractmethod
    def close(self) -> None:
        """Close :class:`~Manager` instance and clear all environments and snapshots."""

    @abc.abstractmethod
    def create(self, environment: typing.Optional[str] = None, *, snapshot: typing.Optional[str] = None) -> str:
        """
        Create and activate new environment.

        :param environment: Optional name of the environment to create.
        :param snapshot: Snapshot to initialize environment from.
        :return: Name of the new environment.
        """

    @abc.abstractmethod
    def deactivate(self) -> None:
        """
        Deactivate current environment.

        Restores to the previous system `HOME` environment.
        """

    @abc.abstractmethod
    def forget(self, snapshot: str) -> None:
        """
        Remove snapshot.

        :param snapshot: Name of the snapshot to remove.
        """

    @abc.abstractmethod
    def remove(self, environment: typing.Optional[str] = None) -> None:
        """
        Remove environment.

        If current environment is being removed - it will be deactivated first.

        :param environment: Optional name of the environment to remove.

                            If not provided: current environment will be deactivated and removed.
        """

    @abc.abstractmethod
    def save(self, snapshot: typing.Optional[str] = None, *, environment: typing.Optional[str] = None) -> str:
        """
        Create a snapshot of a current environment.

        :param snapshot: Optional name of the snapshot.
        :param environment: Optional name of the environment to create snapshot from.

                            If not provided: current environment will be used.
        :return: Name of the new snapshot.
        """


class DirectoryManagerState(typing.NamedTuple):
    """Information about current active environment used by :class:`~DirectoryManager`."""

    environment: str
    backup: typing.Mapping[str, typing.Optional[str]]


class DirectoryManager(Manager):
    """Manager for the temporary fake directories."""

    __slots__ = ('__root', '__state', '__tree')

    def __init__(self, *args: str) -> None:
        """Initialize new :class:`~DirectoryManager` instance."""
        raw: typing.Mapping[str, typing.Sequence[str]] = {
            key: [f'{index:04d}', *value]
            for index, (_, group) in enumerate(itertools.groupby(
                sorted(
                    {arg: split_path(os.environ.get(arg, uuid.uuid4().hex)) for arg in args}.items(),
                    key=lambda item: (item[1], item[0]),
                ),
                lambda item: item[1][0],
            ))
            for key, value in remove_prefixes(dict(group)).items()
        }

        self.__root: typing.Final[str] = tempfile.mkdtemp()
        self.__state: typing.Optional[DirectoryManagerState] = None
        self.__tree: typing.Final[typing.Mapping[str, str]] = {
            key: os.path.join('', *value)
            for key, value in remove_prefixes(raw).items()
        }

    @property
    def current(self) -> typing.Optional[str]:  # noqa: D401
        """Name of the current active home environment."""
        if self.__state is None:
            return None
        return self.__state.environment

    @property
    def environments(self) -> typing.Sequence[str]:  # noqa: D401
        """List of available home environments."""
        return [
            name
            for name in os.listdir(self.__root)
            if os.path.isdir(os.path.join(self.__root, name))
        ]

    @property
    def snapshots(self) -> typing.Sequence[str]:  # noqa: D401
        """List of available snapshots."""
        return [
            name[:-len(SNAPSHOT_EXTENSION)]
            for name in os.listdir(self.__root)
            if os.path.isfile(os.path.join(self.__root, name)) and name.endswith(SNAPSHOT_EXTENSION)
        ]

    def activate(self, environment: str) -> None:
        """
        Activate existing environment.

        :param environment: Name of the environment to activate.
        """
        environment_path: str = os.path.join(self.__root, environment)
        if not os.path.isdir(environment_path):
            raise ValueError('environment does not exist')

        backup: typing.Mapping[str, typing.Optional[str]]
        if self.__state is None:
            backup = {
                key: os.environ.get(key, None)
                for key in self.__tree
            }
        else:
            backup = self.__state.backup

        self.__state = DirectoryManagerState(environment=environment, backup=backup)

        key: str
        value: str
        for key, value in self.__tree.items():
            os.environ[key] = os.path.join(environment_path, value)

    def close(self) -> None:
        """Close :class:`~Manager` instance and clear all environments and snapshots."""
        if self.__state is not None:
            self.deactivate()

        shutil.rmtree(self.__root)

    def create(self, environment: typing.Optional[str] = None, *, snapshot: typing.Optional[str] = None) -> str:
        """
        Create and activate new environment.

        :param environment: Optional name of the environment to create.
        :param snapshot: Snapshot to initialize environment from.
        :return: Name of the new environment.
        """
        if environment is None:
            environment = uuid.uuid4().hex

        environment_path: str = os.path.join(self.__root, environment)
        if os.path.isdir(environment_path):
            raise ValueError('environment already exists')

        os.makedirs(environment_path)

        if snapshot is None:
            directory_name: str
            for directory_name in self.__tree.values():
                os.makedirs(os.path.join(environment_path, directory_name), exist_ok=True)

        else:
            snapshot_path: str = os.path.join(self.__root, snapshot + SNAPSHOT_EXTENSION)
            if not os.path.isfile(snapshot_path):
                raise ValueError('snapshot not found')

            with tarfile.TarFile(snapshot_path, 'r') as snapshot_container:
                snapshot_container.extractall(environment_path)  # nosec

        self.activate(environment)

        return environment

    def deactivate(self) -> None:
        """
        Deactivate current environment.

        Restores to the previous system `HOME` environment.
        """
        if self.__state is None:
            raise ValueError('no active environment found')

        key: str
        value: typing.Optional[str]
        for key, value in self.__state.backup.items():
            if value is None:
                del os.environ[key]
            else:
                os.environ[key] = value

        self.__state = None

    def forget(self, snapshot: str) -> None:
        """
        Remove snapshot.

        :param snapshot: Name of the snapshot to remove.
        """
        snapshot_path: str = os.path.join(self.__root, snapshot + SNAPSHOT_EXTENSION)
        if not os.path.isfile(snapshot_path):
            raise ValueError('snapshot not found')

        os.remove(snapshot_path)

    def remove(self, environment: typing.Optional[str] = None) -> None:
        """
        Remove environment.

        If current environment is being removed - it will be deactivated first.

        :param environment: Optional name of the environment to remove.

                            If not provided: current environment will be deactivated and removed.
        """
        if environment is None:
            if self.__state is None:
                raise ValueError('no active environment found')
            environment = self.__state.environment

        environment_path: str = os.path.join(self.__root, environment)
        if not os.path.isdir(environment_path):
            raise ValueError('environment not found')

        if (self.__state is not None) and (self.__state.environment == environment):
            self.deactivate()

        shutil.rmtree(environment_path)

    def save(self, snapshot: typing.Optional[str] = None, *, environment: typing.Optional[str] = None) -> str:
        """
        Create a snapshot of a current environment.

        :param snapshot: Optional name of the snapshot.
        :param environment: Optional name of the environment to create snapshot from.

                            If not provided: current environment will be used.
        :return: Name of the new snapshot.
        """
        if snapshot is None:
            snapshot = uuid.uuid4().hex

        if environment is None:
            if self.__state is None:
                raise ValueError('no active environment found')
            environment = self.__state.environment

        snapshot_path: str = os.path.join(self.__root, snapshot + SNAPSHOT_EXTENSION)
        if os.path.isfile(snapshot_path):
            raise ValueError('snapshot already exists')

        environment_path: str = os.path.join(self.__root, environment)
        if not os.path.isdir(environment_path):
            raise ValueError('environment not found')

        snapshot_container: tarfile.TarFile
        with tarfile.TarFile(snapshot_path, 'w') as snapshot_container:
            name: str
            for name in os.listdir(environment_path):
                snapshot_container.add(os.path.join(environment_path, name), name)

        return snapshot


class CompositeManager(Manager):
    """Manager for a group of other managers."""

    __slots__ = ('__children',)

    def __init__(self, *args: Manager) -> None:
        """Initialize new :class:`~CompositeManager` instance."""
        self.__children: typing.Final[typing.Sequence[Manager]] = args

    def activate(self, environment: str) -> None:
        """
        Activate existing environment.

        :param environment: Name of the environment to activate.
        """
        manager: Manager
        for manager in self.__children:
            manager.activate(environment)

    def close(self) -> None:
        """Close :class:`~Manager` instance and clear all environments and snapshots."""
        manager: Manager
        for manager in self.__children:
            manager.close()

    def create(self, environment: typing.Optional[str] = None, *, snapshot: typing.Optional[str] = None) -> str:
        """
        Create and activate new environment.

        :param environment: Optional name of the environment to create.
        :param snapshot: Snapshot to initialize environment from.
        :return: Name of the new environment.
        """
        if environment is None:
            environment = uuid.uuid4().hex

        manager: Manager
        for manager in self.__children:
            manager.create(environment, snapshot=snapshot)

        return environment

    def deactivate(self) -> None:
        """
        Deactivate current environment.

        Restores to the previous system `HOME` environment.
        """
        manager: Manager
        for manager in self.__children:
            manager.deactivate()

    def forget(self, snapshot: str) -> None:
        """
        Remove snapshot.

        :param snapshot: Name of the snapshot to remove.
        """
        manager: Manager
        for manager in self.__children:
            manager.forget(snapshot)

    def remove(self, environment: typing.Optional[str] = None) -> None:
        """
        Remove environment.

        If current environment is being removed - it will be deactivated first.

        :param environment: Optional name of the environment to remove.

                            If not provided: current environment will be deactivated and removed.
        """
        manager: Manager
        for manager in self.__children:
            manager.remove(environment)

    def save(self, snapshot: typing.Optional[str] = None, *, environment: typing.Optional[str] = None) -> str:
        """
        Create a snapshot of a current environment.

        :param snapshot: Optional name of the snapshot.
        :param environment: Optional name of the environment to create snapshot from.

                            If not provided: current environment will be used.
        :return: Name of the new snapshot.
        """
        if snapshot is None:
            snapshot = uuid.uuid4().hex

        manager: Manager
        for manager in self.__children:
            manager.save(snapshot, environment=environment)

        return snapshot
