# -*- coding: utf-8 -*-
# pylint: disable=invalid-name,consider-using-f-string,logging-not-lazy,import-error

"""Script display the conda TOS."""

import logging
import os
import sys

from qtpy.QtCore import Qt, QUrl
from qtpy.QtGui import QDesktopServices
from qtpy.QtSvgWidgets import QSvgWidget
from qtpy.QtWidgets import (
    QApplication, QDialog, QVBoxLayout, QLabel, QPushButton, QHBoxLayout, QGridLayout, QFrame, QCheckBox, QSizePolicy
)

from anaconda_navigator.utils import styles

logger = logging.getLogger(__name__)

TERMS_OF_PRIVACY = 'https://legal.anaconda.com/policies/en/?name=privacy-policy'
TERMS_OF_SERVICE = 'https://legal.anaconda.com/policies/en/#terms-of-service'


class TermsOfServiceDialog(QDialog):
    """Dialog to display the terms of service and get user agreement."""

    def __init__(self, *args, **kwargs):  # pylint: disable=too-many-locals,too-many-statements
        """Dialog to display the terms of service and get user agreement."""
        super().__init__(*args, **kwargs)
        self.setWindowTitle('Terms of Services & Privacy Policy Update')
        self.setWindowFlags(Qt.FramelessWindowHint | Qt.Tool | Qt.WindowStaysOnTopHint)
        self.setAttribute(Qt.WA_TranslucentBackground)
        self.setFixedSize(565, 435)

        root_frame = QFrame(self)
        root_frame.setObjectName('widgetFrame')

        content_frame = QFrame()
        content_frame.setObjectName('content_frame')
        content_frame.setFrameShape(QFrame.NoFrame)
        content_frame.setFixedHeight(344)

        # Header
        header_label = QLabel('Terms of Services & Privacy Policy Update')
        header_label.setFixedHeight(45)
        header_label.setObjectName('header_label')

        # Description
        whats_new_label = QLabel(
            '<p style="line-height:20px;letter-spacing: 0">'
            'See what has changed '
            f'<a style="color:#6D5BF6; text-decoration:none; font-weight: bold" href="{TERMS_OF_SERVICE}">here</a>.')

        desc_label = QLabel(
            '<p style="line-height:20px;letter-spacing: 0">'
            'Please accept below in order to take advantage of Navigator’s capabilities to manage and access:'
            '</p>'
        )
        desc_label.setWordWrap(True)
        whats_new_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
        whats_new_label.linkActivated.connect(self.handle_link_click)

        # Grid for icond and features
        grid_layout = QGridLayout()
        grid_layout.setContentsMargins(0, 24, 0, 12)
        features = [
            (os.path.join(styles.images.ICONS_PATH, 'environment_icon.svg'), 'Environments', (40, 32)),
            (os.path.join(styles.images.ICONS_PATH, 'packages_icon.svg'), '300+ Channels<br>& Packages', (40, 32)),
            (os.path.join(styles.images.ICONS_PATH, 'apps_icon.svg'), '25 Data<br>Science Apps', (40, 32)),
            (os.path.join(styles.images.ICONS_PATH, 'courses_icon.svg'), '50 Learning<br>Courses', (40, 32)),
        ]

        for i, (icon_path, feature, dimensions) in enumerate(features):
            icon_label = QSvgWidget(icon_path)
            icon_label.setFixedSize(*dimensions)

            # Add a QLabel for the feature text
            text_label = QLabel(f'<p style="line-height:20px">{feature}</p>')
            text_label.setAlignment(Qt.AlignCenter)
            text_label.setObjectName('feature_label')

            grid_layout.addWidget(icon_label, 0, i, alignment=Qt.AlignCenter)
            grid_layout.addWidget(text_label, 1, i, alignment=Qt.AlignTop)

        notice_label = QLabel('If you do not wish to accept, you will have no access and can close the app.')

        self.checkbox = QCheckBox()
        self.checkbox.checkStateChanged.connect(
            lambda state: self.continue_button.setEnabled(state == Qt.CheckState.Checked)
        )

        checkbox_text = QLabel(
            '<p style="line-height:20px">'
            'I acknowledge that I have read and consent to Anaconda\'s<br> '
            f'<a style="color:#6D5BF6; text-decoration:none; font-weight: bold" href="{TERMS_OF_PRIVACY}">'
            'Privacy Policy</a> and '
            f'<a style="color:#6D5BF6; text-decoration:none; font-weight: bold" href="{TERMS_OF_SERVICE}">'
            'Terms of Service</a>, '
            'including the collection, use, and processing of my personal data as described therein.'
            '</p>'
        )
        checkbox_text.setWordWrap(True)
        checkbox_text.setTextInteractionFlags(Qt.TextBrowserInteraction)
        checkbox_text.linkActivated.connect(self.handle_link_click)
        checkbox_text.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)

        self.continue_button = QPushButton('Continue to Navigator')
        self.continue_button.setEnabled(False)
        self.continue_button.clicked.connect(self.accept_tos)

        checkbox_layout = QHBoxLayout()
        checkbox_layout.setAlignment(Qt.AlignLeft)
        checkbox_layout.setSpacing(0)

        button_layout = QHBoxLayout()
        button_layout.addStretch()
        button_layout.addWidget(self.continue_button)

        checkbox_layout.addWidget(self.checkbox, alignment=Qt.AlignTop)
        checkbox_layout.addWidget(checkbox_text)

        content_layout = QVBoxLayout()
        content_layout.setContentsMargins(16, 8, 16, 24)
        content_layout.addWidget(whats_new_label)
        content_layout.addWidget(desc_label)
        content_layout.addLayout(grid_layout)
        content_layout.addWidget(notice_label)
        content_layout.addLayout(checkbox_layout)

        content_frame.setLayout(content_layout)

        layout = QVBoxLayout()
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)
        layout.setAlignment(Qt.AlignTop)
        layout.addWidget(header_label, alignment=Qt.AlignLeft)
        layout.addWidget(content_frame)
        layout.addLayout(button_layout)
        root_frame.setLayout(layout)

        root_layout = QVBoxLayout()
        root_layout.setContentsMargins(0, 0, 0, 0)
        root_layout.setSpacing(0)
        root_layout.addWidget(root_frame)

        self.setLayout(root_layout)

    def handle_link_click(self, url):
        """Handle the links clicked in the dialog."""
        QDesktopServices.openUrl(QUrl(url))
        logger.info('User clicked on link: %s' % url)

    def accept_tos(self):
        """Handle the 'Continue to Navigator' button click."""
        logger.info('User accepted the terms and proceeded.')
        self.accept()


if __name__ == '__main__':
    app = QApplication(sys.argv)
    dialog = TermsOfServiceDialog()

    dialog.setStyleSheet(styles.load_style_sheet())

    if dialog.exec_():
        logger.info('User accepted the terms and proceeded.')
    else:
        logger.info('User closed the dialog without accepting the terms.')
