"""Repository selector dialog for login reminders."""
from __future__ import annotations

import typing

from qtpy.QtCore import Qt, Signal
from qtpy.QtWidgets import QLabel, QVBoxLayout

from anaconda_navigator.config import AnacondaBrand
from anaconda_navigator.utils.widgets.signals import DialogOutcome, DialogValue
from anaconda_navigator.widgets import ButtonPrimary, ButtonLink, ComboBoxBase
from anaconda_navigator.widgets.dialogs.login import AccountOutcome, AccountValue
from anaconda_navigator.widgets.dialogs.login.reminder.styling import BaseCleanDialog

if typing.TYPE_CHECKING:
    from anaconda_navigator.widgets import main_window


class RepoSelectorDialog(BaseCleanDialog):  # pylint: disable=too-few-public-methods
    """Repository selector dialog."""
    sig_dialog_request = Signal(DialogOutcome, DialogValue)
    sig_login_request = Signal(AccountOutcome, AccountValue)

    def __init__(self, parent: 'main_window.MainWindow', title: str = 'Connect to a Repository'):
        """Initialize new :class:`~RepoSelectorDialog` instance."""
        super().__init__(parent, title)

        self.__message_label = QLabel('Select a repository:')

        self.__repo_combobox = ComboBoxBase()
        self.__repo_combobox.addItem(AnacondaBrand.ANACONDA_ORG, 'navigator://individual/login')
        self.__repo_combobox.addItem(AnacondaBrand.COMMERCIAL_EDITION, 'navigator://commercial/login')
        self.__repo_combobox.addItem(AnacondaBrand.TEAM_EDITION, 'navigator://team/login')
        self.__repo_combobox.addItem(AnacondaBrand.ENTERPRISE_EDITION, 'navigator://enterprise/login')

        self.__repo_login_button = ButtonPrimary('Continue')

        self.__cloud_login_button = ButtonLink('Sign in to Anaconda Cloud')

        self.__content_layout = QVBoxLayout()
        self.__content_layout.setAlignment(Qt.AlignTop)
        self.__content_layout.addWidget(self.__message_label)
        self.__content_layout.addSpacing(24)
        self.__content_layout.addWidget(self.__repo_combobox, alignment=Qt.AlignLeft)
        self.__content_layout.addSpacing(20)
        self.__content_layout.addWidget(self.__repo_login_button, alignment=Qt.AlignLeft)
        self.__content_layout.addSpacing(24)
        self.__content_layout.addWidget(self.__cloud_login_button, alignment=Qt.AlignLeft)

        self.__repo_login_button.clicked.connect(self.__open_link)
        self.__cloud_login_button.clicked.connect(lambda: self.sig_dialog_request.emit(
            DialogOutcome.DIALOG_REQUEST, DialogValue.LOGIN_REMINDER))

        self.sig_dialog_request.connect(self.reject)
        self.sig_login_request.connect(self.accept)

        self.setLayout(self.__content_layout)

    def __open_link(self, link: str | None) -> None:
        """Process link opening action."""
        outcome, value = {
            'navigator://individual/login': (AccountOutcome.LOGIN_REQUEST, AccountValue.INDIVIDUAL_EDITION),
            'navigator://commercial/login': (AccountOutcome.LOGIN_REQUEST, AccountValue.COMMERCIAL_EDITION),
            'navigator://team/login': (AccountOutcome.LOGIN_REQUEST, AccountValue.TEAM_EDITION),
            'navigator://enterprise/login': (AccountOutcome.LOGIN_REQUEST, AccountValue.ENTERPRISE_EDITION),
        }[link or self.__repo_combobox.itemData(self.__repo_combobox.currentIndex())]

        self.sig_login_request.emit(outcome, value)
        self.accept()
