"""
Manage your Anaconda repository channels.
"""

from __future__ import print_function, unicode_literals

import argparse
from typing import Iterable

from tabulate import tabulate

from ..utils.format import format_header_and_rows
from .base import SubCommandBase

fields = [
    "id",
    "curated",
    "score",
    "score_type",
    "description",
    "published_at",
    "packages_count",
]

fields_v2 = [
    "cvssv2.cvssV2.accessComplexity",
    "cvssv2.cvssV2.accessVector",
    "cvssv2.cvssV2.authentication",
    "cvssv2.cvssV2.availabilityImpact",
    "cvssv2.cvssV2.confidentialityImpact",
    "cvssv2.cvssV2.integrityImpact",
    "cvssv2.cvssV2.baseScore",
    "cvssv2.severity",
    "cvssv3.cvssV3.userInteraction",
]

fields_v3 = [
    "cvssv3.cvssV3.attackComplexity",
    "cvssv3.cvssV3.attackVector",
    "cvssv3.cvssV3.availabilityImpact",
    "cvssv3.cvssV3.baseScore",
    "cvssv3.cvssV3.baseSeverity",
    "cvssv3.cvssV3.confidentialityImpact",
    "cvssv3.cvssV3.integrityImpact",
    "cvssv3.cvssV3.privilegesRequired",
    "cvssv3.cvssV3.scope",
]

fields_v4 = [
    "cvssv4.cvssV4.baseScore",
    "cvssv4.cvssV4.attackVector",
    "cvssv4.cvssV4.attackComplexity",
    "cvssv4.cvssV4.privilegesRequired",
    "cvssv4.cvssV4.userInteraction",
    "cvssv4.cvssV4.baseSeverity",
    "cvssv4.cvssV4.vulnerableSystemIntegrity",
    "cvssv4.cvssV4.vulnerableSystemConfidentiality",
    "cvssv4.cvssV4.vulnerableSystemAvailability",
    "cvssv4.cvssV4.vectorString",
]

show_all_fields = [
    "id",
    "score",
    "score_type",
    "curated",
    "packages_count",
    "description",
]


keymap = {
    "id": "CVE ID",
    "score_type": "Type",
    "packages_count": "# Packages",
    "cvssv2.cvssV2.accessComplexity": "CVSS V2 Access Complexity",
    "cvssv2.cvssV2.accessVector": "CVSS V2 Access Vector",
    "cvssv2.cvssV2.authentication": "CVSS V2 Authentication",
    "cvssv2.cvssV2.availabilityImpact": "CVSS V2 Availablity Impact",
    "cvssv2.cvssV2.confidentialityImpact": "CVSS V2 Confidentiality Impact",
    "cvssv2.cvssV2.integrityImpact": "CVSS V2 Integrity Impact",
    "cvssv2.cvssV2.baseScore": "CVSS V2 Base Score",
    "cvssv2.severity": "CVSS V2 Severity",
    "cvssv3.cvssV3.attackComplexity": "CVSS V3 Attack Complexity",
    "cvssv3.cvssV3.attackVector": "CVSS V3 Attack Vector",
    "cvssv3.cvssV3.availabilityImpact": "CVSS V3 Availability Impact",
    "cvssv3.cvssV3.baseScore": "CVSS V3 Base Score",
    "cvssv3.cvssV3.baseSeverity": "CVSS V3 Base Severity",
    "cvssv3.cvssV3.confidentialityImpact": "CVSS V3 Confidentiality Impact",
    "cvssv3.cvssV3.integrityImpact": "CVSS V3 Integrity Impact",
    "cvssv3.cvssV3.privilegesRequired": "CVSS V3 Privileges Required",
    "cvssv3.cvssV3.scope": "CVSS V3 Scope",
    "cvssv3.cvssV3.userInteraction": "CVSS V3 User Interaction",
    "cvssv4.cvssV4.baseScore": "CVSS V4 Base Score",
    "cvssv4.cvssV4.baseSeverity": "CVSS V4 Base Severity",
    "cvssv4.cvssV4.attackVector": "CVSS V4 Attack Vector",
    "cvssv4.cvssV4.attackComplexity": "CVSS V4 Attack Complexity",
    "cvssv4.cvssV4.privilegesRequired": "CVSS V4 Privileges Required",
    "cvssv4.cvssV4.userInteraction": "CVSS V4 User Interaction",
    "cvssv4.cvssV4.baseSeverity": "CVSS V4 Exploitability",
    "cvssv4.cvssV4.vulnerableSystemIntegrity": "CVSS V4 Impact",
    "cvssv4.cvssV4.vulnerableSystemConfidentiality": "CVSS V4 Confidentiality",
    "cvssv4.cvssV4.vulnerableSystemAvailability": "CVSS V4 Availability",
    "cvssv4.cvssV4.vectorString": "CVSS V4 Vector String",
}


class SubCommand(SubCommandBase):
    name = "cves"

    def main(self):
        self.log.info("")
        args = self.args
        if args.list:
            self.show_list(args.offset, args.limit)
        elif args.show:
            self.show(args.show)
        elif args.show_files:
            self.show_files(args.show_files, args.offset, args.limit)
        else:
            raise NotImplementedError()

    def show_list(self, offset=0, limit=20):
        data = self.api.get_cves(offset, limit)

        list_data = []
        for cve in data["items"]:
            cve = self.normalize_item(cve)
            cve = dict((k, cve.get(k, "")) for k in show_all_fields)
            # Normalize None values to Empty because of Tabulate issue: https://github.com/astanin/python-tabulate/issues/312
            cve = {k: (v if v is not None else "") for k, v in cve.items()}
            cve["description"] = cve["description"][:100]
            list_data.append(cve)

        self.log.info(tabulate(list_data, tablefmt="grid", maxcolwidths=[30, 30]))
        self.log.info("")

    def show(self, cve):
        data = self.api.get_cve(cve)

        def show_fields(headline, fields):
            self.log.info(headline)
            self.log.info(
                tabulate(
                    self.format_detail(data, fields),
                    tablefmt="grid",
                    maxcolwidths=[None, 30],
                )
            )
            self.log.info("")

        show_fields(f"Showing details for {cve}:", fields)
        show_fields("CVSS V2:", fields_v2)
        show_fields("CVSS V3:", fields_v3)
        show_fields("CVSS V4:", fields_v4)

    def show_files(self, cve_id, offset=0, limit=50):
        data = self.api.get_cve_files(cve_id, offset, limit)
        self.log.info(
            "Showing %cve packages, associated with %s, starting from %cve from total %cve:"
            % (limit, cve_id, offset, data["total_count"])
        )

        rows, header = format_header_and_rows(
            data["items"],
            [
                "blob_sha256",
                "blob_size",
                "cve_status",
                "parent_channel_name",
                "artifact_type",
            ],
        )
        self.log.info(tabulate(rows, headers=header, tablefmt="grid"))
        self.log.info("")

    def format_detail(self, item, fields=fields):
        item_ = self.normalize_item(item)
        resp = []

        for key in fields:
            label = keymap.get(key, key).capitalize()
            value = self.nested_get(item_, key, "")
            resp.append([label, value])

        return resp

    def nested_get(self, item, path, default=None):
        """Returns the value of the nested dictionary, using path.

        Example:
            nested_get({"a":{"b": 42}}, "a.b") == 42
        """
        result = item
        for key in path.split("."):
            if not isinstance(result, Iterable) or key not in result:
                return default
            result = result[key]
        return result

    @staticmethod
    def normalize_item(item):
        item_ = item.copy()
        score = item_.get("cvssv4_score")
        if score:
            item_["score"] = score
            item_["score_type"] = "CVSS4"
        elif item_.get("cvssv3_score"):
            item_["score"] = item_.get("cvssv3_score")
            item_["score_type"] = "CVSS3"
        else:
            item_["score"] = item_.get("cvssv2_score")
            item_["score_type"] = "CVSS2"

        def fmt_package(pack):
            try:
                if "subdir" not in pack:
                    pack["subdir"] = ""
                return "{subdir}/{name}-{version} (sha258: {sha256})".format(**pack)
            except TypeError:
                return pack

        packages = [fmt_package(pack) for pack in item_.get("packages", [])]
        item_["packages"] = ", ".join(packages)
        return item_

    def add_parser(self, subparsers):
        subparser = subparsers.add_parser(
            self.name,
            help="Access Anaconda Repository {}s".format(self.name),
            formatter_class=argparse.RawDescriptionHelpFormatter,
            description=__doc__,
        )

        group = subparser.add_mutually_exclusive_group(required=True)

        group.add_argument(
            "--list",
            action="store_true",
            help="list all {}s for a user".format(self.name),
        )
        group.add_argument(
            "--show",
            metavar=self.name.upper(),
            help="Show details about {}".format(self.name),
        )
        group.add_argument(
            "--show-files",
            metavar=self.name.upper(),
            help="Show files for {} (Use limit/offset additionally)".format(self.name),
        )

        subparser.add_argument(
            "-o",
            "--offset",
            default=0,
            type=int,
            help="Offset when displaying the results",
        )
        subparser.add_argument(
            "-l",
            "--limit",
            default=20,
            type=int,
            help="Offset when displaying the results",
        )

        subparser.set_defaults(main=self.main)
