from __future__ import unicode_literals

import argparse

INITIAL_SPACE = "     "
fmt_package_headers = (
    INITIAL_SPACE + "%(channel_path)-30s | %(name)-40s | %(version)8s | %(family)-12s "
    "| %(build_number)-10s | %(license)-15s | %(subdirs)-15s"
)

fmt_package_simple_headers = (
    INITIAL_SPACE + "%(name)-40s | %(file_count)-12s | %(download_count)-13s "
    "| %(license)-15s | %(family)-12s"
)
fmt_channel_headers = (
    INITIAL_SPACE
    + "%(channel_path)-30s | %(privacy)-10s | %(owners)15s | %(artifact_count)-12s "
    "| %(download_count)-11s | %(subchannel_count)-14s |  %(mirror_count)-9s | %(description)-30s"
)

fmt_sync_state_header_spacer = {
    "active": "-" * 10,
    "failed": "-" * 10,
    "passive": "-" * 10,
    "removed": "-" * 10,
    "last_ckey": "-" * 60,
    "last_exception": "-" * 18,
    "total_steps_count": "-" * 18,
    "current_step_number": "-" * 20,
    "current_step_percentage": "-" * 25,
    "current_step_description": "-" * 34,
}
fmt_sync_state_headers = (
    INITIAL_SPACE
    + "%(active)-10s | %(failed)-10s | %(passive)-10s | %(removed)-10s | %(last_ckey)-60s  | %(last_exception)-18s  "
    "| %(total_steps_count)-18s | %(current_step_number)-20s | %(current_step_percentage)-25s | %(current_step_description)-34s"
)

fmt_sync_state_mirror_filtered_header_spacer = {
    "cve": "-" * 18,
    "date": "-" * 18,
    "license": "-" * 18,
    "only_specs": "-" * 18,
    "exclude_specs": "-" * 18,
    "duplicated_legacy_conda": "-" * 18,
}
fmt_sync_state_mirror_filtered_header = (
    INITIAL_SPACE
    + "%(cve)-18s | %(date)-18s | %(license)-18s | %(only_specs)-18s | %(exclude_specs)-18s  | %(duplicated_legacy_conda)-18s"
)

fmt_sync_state_channel_filtered_header_spacer = {
    "license": "-" * 18,
    "exclude_specs": "-" * 18,
}
fmt_sync_state_channel_filtered_header = (
    INITIAL_SPACE + "%(license)-18s | %(exclude_specs)-18s"
)


class PackagesFormatter:
    def __init__(self, log):
        self.log = log

    @staticmethod
    def format_package_header(simple=False):
        if simple:
            package_header = {
                "name": "Name",
                "file_count": "# of files",
                "download_count": "# of download",
                "license": "License",
                "family": "Artifact Family",
            }
            fmt = fmt_package_simple_headers
        else:
            package_header = {
                "channel_path": "Channel",
                "name": "Name",
                "family": "Artifact Family",
                "version": "Version",
                "subdirs": "Platforms",
                "license": "License",
                "build_number": "Build",
            }
            fmt = fmt_package_headers
        return fmt % package_header

    def log_format_package_header(self, simple=False):
        self.log.info(self.format_package_header(simple=simple))

    @staticmethod
    def format_package(package, simple=False):
        package = package.copy()
        package.update(package["metadata"])

        if package["subchannel"]:
            package["channel_path"] = "%s/%s" % (
                package["channel"],
                package["subchannel"],
            )
        else:
            package["channel_path"] = package["channel"]

        if "subdirs" not in package:
            package["subdirs"] = []

        if "build_number" not in package:
            package["build_number"] = ""

        if "license" not in package:
            package["license"] = ""

        if "family" not in package:
            package["family"] = ""

        if "version" not in package:
            package["version"] = ""

        package["full_name"] = "%s::%s" % (package["channel_path"], package["name"])
        package["subdirs"] = ", ".join(
            str(x) for x in package["subdirs"] if x is not None
        )

        if simple:
            fmt = fmt_package_simple_headers
        else:
            fmt = fmt_package_headers

        return fmt % package

    def log_format_package(self, package, simple=False):
        self.log.info(self.format_package(package, simple=simple))

    @staticmethod
    def format_channel_header():
        package_header = {
            "channel_path": "Channel",
            "privacy": "Privacy",
            "owners": "Owners",
            "artifact_count": "# Artifacts",
            "download_count": "# Downloads",
            "subchannel_count": "# Subchannels",
            "mirror_count": "# Mirrors",
            "description": "Description",
        }
        return fmt_channel_headers % package_header

    def log_format_channel_header(self):
        self.log.info(self.format_channel_header())

    @staticmethod
    def format_channel(channel):
        channel = channel.copy()

        if channel["parent"]:
            channel["channel_path"] = "%s/%s" % (channel["parent"], channel["name"])
        else:
            channel["channel_path"] = channel["name"]

        channel["owners"] = ", ".join(
            str(x) for x in channel["owners"] if x is not None
        )
        return fmt_channel_headers % channel

    def log_format_channel(self, channel):
        self.log.info(self.format_channel(channel))

    def format(self, packages, metadata, simple=False):
        self.log_format_package_header(simple=simple)

        if simple:
            package_header = {
                "name": "-" * 40,
                "file_count": "-" * 8,
                "download_count": "-" * 13,
                "license": "-" * 15,
                "family": "-" * 12,
            }

            self.log.info(fmt_package_simple_headers % package_header)
        else:
            package_header = {
                "channel_path": "-" * 30,
                "name": "-" * 40,
                "version": "-" * 8,
                "family": "-" * 12,
                "subdirs": "-" * 15,
                "license": "-" * 15,
                "build_number": "-" * 10,
            }

            self.log.info(fmt_package_headers % package_header)

        for package in packages:
            self.log_format_package(package, simple=simple)

        if packages:
            offset = metadata.get("offset") or 0
            end_set = len(packages) + offset
            self.log.info(
                "\n%s%i packages found." % (INITIAL_SPACE, metadata["total_count"])
            )
            self.log.info(
                "%sVisualizing %i-%i interval." % (INITIAL_SPACE, offset, end_set)
            )
        else:
            self.log.info("No packages found")

        self.log.info("")


class SyncStateFormatter:
    keymap = {
        "last_ckey": "Last ckey",
        "last_exception": "Last exception",
        "total_steps_count": "Total steps count",
        "current_step_number": "Current step number",
        "current_step_percentage": "Current step percentage",
        "current_step_description": "Current step description",
    }
    keymap_mirror = {
        "only_specs": "Only specs",
        "exclude_specs": "exclude specs",
        "duplicated_legacy_conda": "Duplicated legacy conda",
    }
    keymap_channel = {"exclude_specs": "Exclude specs"}

    @staticmethod
    def format_sync_state(data, mirror_name):
        mirror = None
        for m in data:
            if m["name"] == mirror_name:
                mirror = m

        if not mirror:
            lines = []
            lines.append("No Mirror named %s" % mirror_name)
            return "\n".join(lines)

        sync_state = mirror["sync_state"]

        lines = []

        lines.append(INITIAL_SPACE + "SYNC STATE:")
        lines.append(SyncStateFormatter.format_sync_state_headers())
        lines.append(fmt_sync_state_headers % fmt_sync_state_header_spacer)
        lines.append(SyncStateFormatter.format_sync_state_general(sync_state))
        lines.append("")

        lines.append(INITIAL_SPACE + "FILTERED PACKAGES BY MIRROR:")
        lines.append(SyncStateFormatter.format_filtered_mirror_headers())
        lines.append(
            fmt_sync_state_mirror_filtered_header
            % fmt_sync_state_mirror_filtered_header_spacer
        )
        lines.append(
            SyncStateFormatter.format_sync_state_filtered_mirror(sync_state["count"])
        )
        lines.append("")

        lines.append(INITIAL_SPACE + "FILTERED PACKAGES BY CHANNEL:")
        lines.append(SyncStateFormatter.format_filtered_channel_headers())
        lines.append(
            fmt_sync_state_channel_filtered_header
            % fmt_sync_state_channel_filtered_header_spacer
        )
        lines.append(
            SyncStateFormatter.format_sync_state_filtered_channel(sync_state["count"])
        )
        lines.append("")

        return "\n".join(lines)

    @classmethod
    def format_sync_state_headers(cls):
        sync_state_headers = {k: k.capitalize() for k in fmt_sync_state_header_spacer}
        sync_state_headers.update(cls.keymap)
        return fmt_sync_state_headers % sync_state_headers

    @staticmethod
    def format_sync_state_general(sync_state):
        sync_state = sync_state.copy()
        count = sync_state["count"]
        sync_state.pop("count", None)
        sync_state.update(count)
        sync_state.pop("mirror_filtered", None)
        sync_state.pop("channel_filtered", None)

        return fmt_sync_state_headers % sync_state

    @classmethod
    def format_filtered_mirror_headers(cls):
        sync_state_mirror_filtered_header = {
            k: k.capitalize() for k in fmt_sync_state_mirror_filtered_header_spacer
        }
        sync_state_mirror_filtered_header.update(cls.keymap_mirror)
        return fmt_sync_state_mirror_filtered_header % sync_state_mirror_filtered_header

    @staticmethod
    def format_sync_state_filtered_mirror(sync_state):
        mirror_filtered = sync_state["mirror_filtered"]
        return fmt_sync_state_mirror_filtered_header % mirror_filtered

    @classmethod
    def format_filtered_channel_headers(cls):
        sync_state_channel_filtered_header = {
            k: k.capitalize() for k in fmt_sync_state_channel_filtered_header_spacer
        }
        sync_state_channel_filtered_header.update(cls.keymap_channel)
        return (
            fmt_sync_state_channel_filtered_header % sync_state_channel_filtered_header
        )

    @staticmethod
    def format_sync_state_filtered_channel(sync_state):
        channel_filtered = sync_state["channel_filtered"]
        return fmt_sync_state_channel_filtered_header % channel_filtered


class FormatterBase:
    entity = ""
    fmt_header_spacer = {}
    fmt_headers = INITIAL_SPACE + ""

    keymap = {
        "date_added": "Published",
        "updated_at": "Updated at",
        "last_run_at": "Last run at",
    }

    fields = []

    @classmethod
    def format_detail(cls, item):
        item_ = cls.normalize_item(item)
        resp = [
            INITIAL_SPACE + "%s Details:" % cls.entity,
            INITIAL_SPACE + "---------------",
        ]

        for key in cls.fields:
            label = cls.keymap.get(key, key).capitalize()
            value = nested_get(item_, key, "")
            resp.append("%s%s: %s" % (INITIAL_SPACE, label, value))

        return "\n".join(resp)

    @classmethod
    def format_list_headers(cls):
        list_headers = {k: k.capitalize() for k in cls.fmt_header_spacer}
        list_headers.update(cls.keymap)
        return cls.fmt_headers % list_headers

    @classmethod
    def format_list_item(cls, item):
        item_ = cls.normalize_item(item)
        return cls.fmt_headers % item_

    @staticmethod
    def normalize_item(item):
        raise NotImplementedError

    @classmethod
    def format_list(cls, items):
        lines = []
        lines.append(cls.format_list_headers())
        lines.append(cls.fmt_headers % cls.fmt_header_spacer)

        for item in items:
            lines.append(cls.format_list_item(item))

        return "\n".join(lines)


class PackageFilesFormatter(FormatterBase):
    entity = "Package Files"
    fmt_header_spacer = {
        "ckey": "-" * 60,
        "name": "-" * 25,
        "version": "-" * 10,
        "platform": "-" * 10,
        "family": "-" * 12,
    }
    fmt_headers = (
        INITIAL_SPACE
        + "%(ckey)-60s | %(name)-25s | %(version)-10s | %(platform)-10s | %(family)-12s"
    )

    keymap = {
        "ckey": "Path",
        "name": "Name",
        "version": "Version",
        "platform": "Platform",
        "family": "Artifact Family",
    }

    @staticmethod
    def normalize_item(item):
        return item


class PackageFilesFormatterWithCVE(FormatterBase):
    entity = "Package Files"
    fmt_header_spacer = {
        "ckey": "-" * 60,
        "name": "-" * 25,
        "cve_score": "-" * 9,
        "cve_status": "-" * 10,
        "version": "-" * 10,
        "platform": "-" * 10,
        "family": "-" * 12,
    }
    fmt_headers = (
        INITIAL_SPACE
        + "%(ckey)-60s | %(name)-25s | %(cve_score)-9s | %(cve_status)-10s | %(version)-10s | %(platform)-10s | %(family)-12s"
    )

    keymap = {
        "ckey": "Path",
        "name": "Name",
        "cve_score": "CVE Score",
        "cve_status": "CVE Status",
        "family": "Artifact Family",
    }

    @staticmethod
    def normalize_item(item):
        item_ = {key: val for key, val in item.items()}
        if item_["cve_status"] is None:
            item_["cve_status"] = "n/a"
        if item_["cve_score"] is None:
            item_["cve_score"] = "n/a"
        return item_


class SettingsFormatter(FormatterBase):
    entity = "Admin Settings"
    fmt_header_spacer = {
        "key": "-" * 36,
        "value": "-" * 12,
    }
    fmt_headers = INITIAL_SPACE + "%(key)-36s | %(value)-12s"

    keymap = {"key": "Setting Name", "value": "Value"}
    fields = ["key", "value"]

    @classmethod
    def normalize_item(cls, item, short_summary=True):
        return item

    @classmethod
    def format_object_as_list(cls, obj):
        return cls.format_list([{"key": k, "value": v} for k, v in obj.items()])


def format_packages(packages, meta, logger):
    formatter = PackagesFormatter(logger)
    formatter.format(packages, meta)
    return formatter


def nested_get(item, path, default=None):
    """Returns the value of the nested dictionary, using path.

    Example:
        nested_get({"a":{"b": 42}}, "a.b") == 42
    """
    result = item
    for key in path.split("."):
        if key not in result:
            return default
        result = result[key]
    return result


def comma_string_to_list(s):
    """Returns a list of strings from the string, splitting by comma and removing trailing empty
        characters.

    Example:
        comma_string_to_list("flask, numpy ") == ["flask", "numpy"]
    """
    return [item.strip() for item in s.split(",") if item.strip()]


def string_to_bool(v):
    if isinstance(v, bool):
        return v
    if v.lower() in ("yes", "true", "t", "y", "1"):
        return True
    elif v.lower() in ("no", "false", "f", "n", "0"):
        return False
    else:
        raise argparse.ArgumentTypeError("Boolean value expected.")


# get header from first item and rows and from data
def format_header_and_rows(data, remove_token_keys=[]):
    rows = [
        [value for key, value in item.items() if key not in remove_token_keys]
        for item in data
    ]
    header = [key for key in data[0].keys() if key not in remove_token_keys]

    return rows, header


# remove milliseconds from dates
def format_dates_remove_ms(data, date_items):
    for item in data:
        for date_item in date_items:
            if item.get(date_item):
                item[date_item] = item[date_item].split(".")[0]
    return data
