import argparse
import logging
from types import SimpleNamespace
from unittest.mock import patch

from repo_cli.commands.policy import SubCommand

POLICY_WITH_DATA = {
    "id": "POLICY_ID",
    "created_at": "2024-09-17T12:45:02.454849+00:00",
    "updated_at": "2024-09-17T12:45:02.454849+00:00",
    "org_id": None,
    "name": "POLICY_WITH_DATA",
    "description": "",
    "schema_version": 2,
    "artifact_family": "conda",
    "policy_state": "unassigned",
    "channel_names": [],
    "mirror_names": [],
    "filters": {
        "add_packages": {
            "type": "compound",
            "operator": "and",
            "subfilters": [
                {
                    "type": "simple",
                    "facet": "platform",
                    "comparator": "in",
                    "criterion": [],
                },
                {
                    "type": "simple",
                    "facet": "license_family",
                    "comparator": "in",
                    "criterion": ["AGPL"],
                },
                {
                    "type": "simple",
                    "facet": "conda_spec",
                    "comparator": "in",
                    "criterion": ["conda"],
                },
                {
                    "type": "simple",
                    "facet": "include_dependencies",
                    "comparator": "==",
                    "criterion": True,
                },
                {
                    "type": "simple",
                    "facet": "only_signed",
                    "comparator": "==",
                    "criterion": False,
                },
                {
                    "type": "simple",
                    "facet": "keep_legacy",
                    "comparator": "==",
                    "criterion": False,
                },
                {
                    "type": "simple",
                    "facet": "package_from_date",
                    "comparator": ">=",
                    "criterion": None,
                },
                {
                    "type": "simple",
                    "facet": "package_to_date",
                    "comparator": "<=",
                    "criterion": None,
                },
                {
                    "type": "compound",
                    "operator": "and",
                    "subfilters": [
                        {
                            "type": "simple",
                            "facet": "cve_score",
                            "comparator": "<",
                            "criterion": 2,
                        },
                        {
                            "type": "simple",
                            "facet": "cve_status",
                            "comparator": "in",
                            "criterion": [],
                        },
                        {
                            "type": "simple",
                            "facet": "cve_allowlist",
                            "comparator": "in",
                            "criterion": [],
                        },
                    ],
                },
                {
                    "type": "simple",
                    "facet": "platform",
                    "comparator": "not in",
                    "criterion": ["win-32"],
                },
            ],
        },
        "remove_packages": {
            "type": "compound",
            "operator": "or",
            "subfilters": [
                {
                    "type": "compound",
                    "operator": "and",
                    "subfilters": [
                        {
                            "type": "simple",
                            "facet": "conda_spec",
                            "comparator": "in",
                            "criterion": ["flask", "123214 ewqewq"],
                        },
                        {
                            "type": "simple",
                            "facet": "conda_spec_exception",
                            "comparator": "in",
                            "criterion": ["1234", "23324"],
                        },
                    ],
                },
                {
                    "type": "simple",
                    "facet": "cve_status",
                    "comparator": "in",
                    "criterion": ["reported", "mitigated", "cleared"],
                },
            ],
        },
    },
}


def test_policy_list_without_data(caplog):
    with patch("repo_cli.commands.base.SubCommandBase") as mock_command_base:
        mock_command_base.api.get_all_policies.return_value = {"items": []}

        mock_command_base.log = logging.getLogger("repo_cli")

        with caplog.at_level(logging.INFO, logger="repo_cli"):
            subcommand = SubCommand(mock_command_base)

            subcommand.list_policies()
            record = caplog.records

            assert len(record) == 1
            assert (
                "| name   | id   | created_at   | updated_at   | artifact_family   | policy_state   | channel_names   |\n+========+======+==============+==============+===================+================+=================+\n+--------+------+--------------+--------------+-------------------+----------------+-----------------+"
                in caplog.text
            )


@patch("repo_cli.commands.base.SubCommandBase")
def test_policy_list_with_data(mock_command_base, caplog):
    mock_command_base.api.get_all_policies.return_value = {"items": [POLICY_WITH_DATA]}

    mock_command_base.log = logging.getLogger("repo_cli")

    with caplog.at_level(logging.INFO, logger="repo_cli"):
        subcommand = SubCommand(mock_command_base)

        subcommand.list_policies()
        record = caplog.records

        assert len(record) == 1
        assert "POLICY_WITH_DATA" in caplog.text
        assert "POLICY_ID" in caplog.text


@patch("repo_cli.commands.base.SubCommandBase")
def test_policy_show(mock_command_base, caplog):
    mock_command_base.api.get_policy.return_value = POLICY_WITH_DATA

    mock_command_base.log = logging.getLogger("repo_cli")

    with caplog.at_level(logging.INFO, logger="repo_cli"):
        subcommand = SubCommand(mock_command_base)

        subcommand.show("POLICY_ID")

        # check for some attributes in show
        assert "| name            | POLICY_WITH_DATA                 |" in caplog.text
        assert "| simple | conda_spec           | in     | ['conda']  |" in caplog.text


@patch("repo_cli.commands.base.SubCommandBase")
def test_policy_create(mock_command_base, caplog):
    mock_command_base.log = logging.getLogger("repo_cli")
    subcommand = SubCommand(mock_command_base)
    subcommand.api.add_policy.return_value = {
        "name": "POLICY_NAME",
        "description": "conda",
        "id": "12345",
    }

    add_package_rules = {}
    remove_package_rules = {}

    subcommand.create_policy(
        "POLICY_NAME",
        "conda",
        add_package_rules=add_package_rules,
        remove_package_rules=remove_package_rules,
    )

    assert "Policy: POLICY_NAME, ID: 12345 created successfully." in caplog.text


@patch("repo_cli.commands.base.SubCommandBase")
def test_policy_edit_empty(mock_command_base, caplog):
    mock_command_base.log = logging.getLogger("repo_cli")
    subcommand = SubCommand(mock_command_base)

    args = argparse.Namespace(
        platform=None,
        platform_not=None,
        license=None,
        license_not=None,
        package_name=None,
        conda_spec=None,
        include_dependencies=None,
        exclude_dependencies=None,
        only_signed_packages=None,
        keep_legacy_packages=None,
        package_created_from=None,
        package_created_to=None,
        cve_link_status_and_score=None,
        cve_score=None,
        cve_status=None,
        cve_allowlist=None,
        conda_spec_exception=None,
        exclude_cve_status=None,
        exclude_package_name=None,
        exclude_package_name_exception=None,
    )

    subcommand.edit_policy("POLICY_ID", "POLICY_NAME", "conda", args=args)

    assert "Policy with id POLICY_ID edited" in caplog.text
