import logging
import re

from repo_cli.utils.constant import CVE_STATUS, LICENSE, PLATFORMS

logger = logging.getLogger("repo_cli")


def parse_and_check_set(input_string, SET, msg):
    # Handle empty or whitespace-only strings
    if not input_string.strip():
        return []
    parsed_set = input_string.split(",")
    if not set(parsed_set).issubset(set(SET)):
        logger.error(msg + ",".join(set(parsed_set) - set(SET)))
        return False
    return parsed_set


def create_filter(criterion, facet, comparator="in", type="simple"):
    return {
        "comparator": comparator,
        "type": type,
        "facet": facet,
        "criterion": criterion,
    }


def validate_cve_id(cve):
    pattern = r"CVE-\d{4}-\d{4}"
    return bool(re.match(pattern, cve))


def add_package_rules(args):
    add_packages = []

    if args.platform and args.platform_not:
        logger.error(
            "You can specify either --platform or --platform-not filter, but not both"
        )
        return False
    if args.license and args.license_not:
        logger.error(
            "You can specify either --license or --license-not filter, but not both"
        )
        return False

    # platform filter
    if args.platform:
        platforms = parse_and_check_set(
            args.platform, PLATFORMS, "Invalid Platforms : "
        )
        if not platforms:
            return False
        add_packages.append(create_filter(platforms, "platform"))

    if args.platform_not:
        platforms = parse_and_check_set(
            args.platform_not, PLATFORMS, "Invalid Platforms : "
        )
        if not platforms:
            return False
        add_packages.append(create_filter(platforms, "platform", comparator="not in"))

    if not (args.platform or args.platform_not):
        add_packages.append(create_filter([], "platform"))

    # license filter
    if args.license:
        license = parse_and_check_set(args.license, LICENSE, "Invalid License : ")
        if not license:
            return False
        add_packages.append(create_filter(license, "license_family"))

    if args.license_not:
        if args.license_not:
            license = parse_and_check_set(
                args.license_not, LICENSE, "Invalid License : "
            )
            if not license:
                return False
            add_packages.append(
                create_filter(license, "license_family", comparator="not in")
            )

    if not (args.license or args.license_not):
        add_packages.append(create_filter([], "license_family"))

    # package name filter
    if args.package_name:
        packages = args.package_name.split(",")
        add_packages.append(create_filter(packages, "conda_spec"))
    else:
        add_packages.append(create_filter([], "conda_spec"))

    # include dependencies
    include_dependencies = args.include_dependencies or (
        args.include_dependencies and not args.exclude_dependencies
    )

    if include_dependencies and not args.package_name:
        logger.error("You need to specify package names to include dependencies.")
        return
    add_packages.append(
        create_filter(include_dependencies, "include_dependencies", comparator="==")
    )

    # only signed packages
    add_packages.append(
        create_filter(bool(args.only_signed_packages), "only_signed", comparator="==")
    )
    # legacy packages
    add_packages.append(
        create_filter(bool(args.keep_legacy_packages), "keep_legacy", comparator="==")
    )

    # package created date filter
    add_packages.append(
        create_filter(
            args.package_created_from if args.package_created_from else None,
            "package_from_date",
            comparator=">=",
        )
    )
    add_packages.append(
        create_filter(
            args.package_created_to if args.package_created_to else None,
            "package_to_date",
            comparator="<=",
        )
    )

    # CVE filter
    cve_subfilters = []

    # cve_score
    cve_score = args.cve_score if args.cve_score else None
    cve_subfilters.append(
        create_filter(cve_score, "cve_score", comparator=args.cve_score_comparator)
    )

    # cve status
    cve_status = args.cve_status if args.cve_status else []
    if args.cve_status:
        cve_status = parse_and_check_set(
            args.cve_status, CVE_STATUS, "Invalid cve status : "
        )
        if not cve_status:
            return False
    cve_subfilters.append(create_filter(cve_status, "cve_status"))

    # cve allowlist
    cve_allowlist = []
    if args.cve_allowlist:
        cve_allowlist = args.cve_allowlist.split(",")
        for cve in cve_allowlist:
            if not validate_cve_id(cve):
                logger.error(
                    f"Invalid CVE id : {cve} - should be in the format CVE-NNNN-NNNN"
                )
                return False
    cve_subfilters.append(create_filter(cve_allowlist, "cve_allowlist"))

    cve_compound = {
        "operator": args.cve_link_status_and_score,
        "type": "compound",
        "subfilters": cve_subfilters,
    }

    add_packages.append(cve_compound)

    return add_packages


def remove_package_rules(args):
    remove_packages = []

    # exclude cve status
    cve_status = []
    if args.exclude_cve_status:
        cve_status = parse_and_check_set(
            args.exclude_cve_status, CVE_STATUS, "Invalid cve status : "
        )
        if not cve_status:
            return False
    remove_packages.append(create_filter(cve_status, "cve_status"))

    # exclude package names
    if args.exclude_package_name:
        exclude_package_names = args.exclude_package_name.split(",")
        conda_spec = create_filter(exclude_package_names, "conda_spec")
    else:
        conda_spec = create_filter([], "conda_spec")

    # exclude package names exception
    if args.exclude_package_name_exception:
        exclude_package_names_exception = args.exclude_package_name_exception.split(",")
        conda_spec_exception = create_filter(
            exclude_package_names_exception, "conda_spec_exception"
        )
    else:
        conda_spec_exception = create_filter([], "conda_spec_exception")

    remove_packages.append(
        {
            "type": "compound",
            "operator": "or",
            "subfilters": [
                conda_spec,
                conda_spec_exception,
            ],
        }
    )

    return remove_packages
