from orangecanvas.localization.si import plsi, plsi_sz, z_besedo
from orangecanvas.localization import Translator  # pylint: disable=wrong-import-order
_tr = Translator("Orange", "biolab.si", "Orange")
del Translator
import numpy as np

from Orange.data import Table
from Orange.widgets import widget, gui
from Orange.widgets.utils import itemmodels
from Orange.widgets.utils.localization import pl
from Orange.widgets.utils.sql import check_sql_input
from Orange.widgets.utils.widgetpreview import WidgetPreview
from Orange.widgets.widget import Input, Output
from Orange.widgets.utils.annotated_data import (create_annotated_table)


class OWSelectByDataIndex(widget.OWWidget):
    name = _tr.m[1457, "Select by Data Index"]
    description = _tr.m[1458, "Match instances by index from data subset."]
    category = _tr.m[1459, "Transform"]
    icon = "icons/SelectByDataIndex.svg"
    priority = 1112

    class Inputs:
        data = Input(_tr.m[1460, "Data"], Table)
        data_subset = Input(_tr.m[1461, "Data Subset"], Table)

    class Outputs:
        matching_data = Output(_tr.m[1462, "Matching Data"], Table, replaces=["Data"], default=True)
        non_matching_data = Output(_tr.m[1463, "Unmatched Data"], Table)
        # avoiding the default annotated output name (Data), as it was used
        # for Matching Data previously
        annotated_data = Output(_tr.m[1464, "Annotated Data"], Table)

    want_main_area = False
    buttons_area_orientation = None
    resizing_enabled = False

    class Warning(widget.OWWidget.Warning):
        instances_not_matching = widget.Msg(_tr.m[1465, "Input tables do not share any instances."])

    def __init__(self):
        super().__init__()

        self.data = None
        self.data_subset = None

        self.model = itemmodels.VariableListModel()
        self.model_unique_with_id = itemmodels.VariableListModel()
        self.extra_model_unique = itemmodels.VariableListModel()
        self.extra_model_unique_with_id = itemmodels.VariableListModel()

        box = gui.widgetBox(self.controlArea, True)
        gui.label(
            box, self, _tr.m[1466, """
Data rows keep their identity even when some or all original variables
are replaced by variables computed from the original ones.

This widget gets two data tables ("Data" and "Data Subset") that
can be traced back to the same source. It selects all rows from Data
that appear in Data Subset, based on row identity and not actual data.
"""].strip(), box=True)

    @Inputs.data
    @check_sql_input
    def set_data(self, data):
        self.data = data

    @Inputs.data_subset
    @check_sql_input
    def set_data_subset(self, data):
        self.data_subset = data

    def handleNewSignals(self):
        self._invalidate()

    def commit(self):
        self.Warning.instances_not_matching.clear()
        subset_ids = []
        if self.data_subset:
            subset_ids = self.data_subset.ids
        if not self.data:
            matching_output = None
            non_matching_output = None
            annotated_output = None
        else:
            if self.data_subset and \
                    not np.intersect1d(subset_ids, self.data.ids).size:
                self.Warning.instances_not_matching()
            row_sel = np.in1d(self.data.ids, subset_ids)
            matching_output = self.data[row_sel]
            non_matching_output = self.data[~row_sel]
            annotated_output = create_annotated_table(self.data, row_sel)

        self.Outputs.matching_data.send(matching_output)
        self.Outputs.non_matching_data.send(non_matching_output)
        self.Outputs.annotated_data.send(annotated_output)

    def _invalidate(self):
        self.commit()

    def send_report(self):
        def data_info_text(data):
            if data is None:
                return _tr.m[1467, "No data."]
            nvars = len(data.domain.variables) + len(data.domain.metas)
            return (f"{data.name}, " + (_tr.e(_tr.c(1468, f"{len(data)} {pl(len(data), 'instance')}, ")) + _tr.e(_tr.c(1469, f"{nvars} {pl(nvars, 'variable')}"))))

        self.report_items("",
                          [(_tr.m[1470, "Data"], data_info_text(self.data)),
                           (_tr.m[1471, "Data Subset"], data_info_text(self.data_subset))])


if __name__ == "__main__":  # pragma: no cover
    iris = Table("iris.tab")
    WidgetPreview(OWSelectByDataIndex).run(
        set_data=iris,
        set_data_subset=iris[:20])
