#
# Copyright (c) 2012-2025 Snowflake Computing Inc. All rights reserved.
#
import threading
import time

from snowflake.snowpark_connect.error.error_codes import ErrorCodes
from snowflake.snowpark_connect.error.error_utils import attach_custom_error_code
from snowflake.snowpark_connect.utils.open_telemetry import otel_create_context_wrapper
from snowflake.snowpark_connect.utils.session import get_or_create_snowpark_session
from snowflake.snowpark_connect.utils.snowpark_connect_logging import logger

_resources_initialized = threading.Event()
_initializer_lock = threading.Lock()
SPARK_VERSION = "3.5.6"
RESOURCE_PATH = "/snowflake/snowpark_connect/resources"
_upload_jars = True  # Flag to control whether to upload jars. Required for Scala UDFs.
_resource_initializer = None  # Will be created lazily when needed


def initialize_resources() -> None:
    """Initialize all expensive resources. We should initialize what we can here, so that actual rpc calls like
    ExecutePlan are as fast as possible."""
    from snowflake.snowpark import functions as snowpark_fn
    from snowflake.snowpark_connect.expression.map_sql_expression import sql_parser

    session = get_or_create_snowpark_session()

    # This could be merged into the sql_parser call, it is done separately because
    # it introduces additional overhead. This finer grained structuring allows us to make finer grained
    # preloading decisions.
    def warm_sql_parser() -> None:
        parser = sql_parser()
        parser.parseExpression("1 + 1")
        parser.parseExpression("CASE WHEN id > 10 THEN 'large' ELSE 'small' END")

    def initialize_session_stage() -> None:
        _ = session.get_session_stage()

    def initialize_catalog() -> None:
        _ = session.catalog

    def warm_up_sf_connection() -> None:
        df = session.create_dataframe([["a", 3], ["b", 2], ["a", 1]], schema=["x", "y"])
        df = df.select(snowpark_fn.upper(df.x).alias("x"), df.y.alias("y2"))
        df = df.group_by(df.x).agg(snowpark_fn.sum("y2"))
        df.collect()

        session.sql("select 1 as sf_connection_warm_up").collect()

    def upload_scala_udf_jars() -> None:
        """Upload Spark jar files required for creating Scala UDFs."""
        stage = session.get_session_stage()
        resource_path = stage + RESOURCE_PATH
        import snowpark_connect_deps_1
        import snowpark_connect_deps_2

        jar_files = [
            f"spark-sql_2.12-{SPARK_VERSION}.jar",
            f"spark-connect-client-jvm_2.12-{SPARK_VERSION}.jar",
            f"spark-common-utils_2.12-{SPARK_VERSION}.jar",
            "sas-scala-udf_2.12-0.1.0.jar",
            "json4s-ast_2.12-3.7.0-M11.jar",
        ]

        for jar_name in jar_files:
            # Try to find the JAR in package 1 first, then package 2
            jar_path = None
            try:
                jar_path = snowpark_connect_deps_1.get_jar_path(jar_name)
            except FileNotFoundError:
                try:
                    jar_path = snowpark_connect_deps_2.get_jar_path(jar_name)
                except FileNotFoundError:
                    raise FileNotFoundError(
                        f"JAR {jar_name} not found in either package"
                    )

            try:
                session.file.put(
                    str(jar_path),
                    resource_path,
                    auto_compress=False,
                    overwrite=False,
                    source_compression="NONE",
                )
            except Exception as e:
                raise RuntimeError(f"Failed to upload JAR {jar_name}: {e}")

    start_time = time.time()

    resources = [
        ("SQL Parser Init", sql_parser),  # Takes about 0.5s
        ("SQL Parser Warm Up", warm_sql_parser),  # Takes about 0.7s
        ("Initialize Session Stage", initialize_session_stage),  # Takes about 0.3s
        ("Initialize Session Catalog", initialize_catalog),  # Takes about 1.2s
        ("Snowflake Connection Warm Up", warm_up_sf_connection),  # Takes about 1s
    ]

    if _upload_jars:
        resources.append(("Upload Scala UDF Jars", upload_scala_udf_jars))

    for name, resource_func in resources:
        resource_start = time.time()
        try:
            resource_func()
            logger.info(f"Initialized {name} in {time.time() - resource_start:.2f}s")
        except Exception as e:
            # We will only log the error if it isn't caused by session being closed. Session
            # closed error happens when the particular run finishes very quickly.
            if str(e).find("because the session has been closed") == -1:
                logger.error(f"Failed to initialize {name}: {e}")

    _resources_initialized.set()
    logger.info(f"All resources initialized in {time.time() - start_time:.2f}s")


def initialize_resources_async() -> threading.Thread:
    """Start resource initialization in background."""
    global _resource_initializer

    with _initializer_lock:
        # Create the thread lazily when needed, capturing current context
        if _resource_initializer is None:
            _resource_initializer = threading.Thread(
                target=otel_create_context_wrapper(initialize_resources),
                name="ResourceInitializer",
            )

        if not _resource_initializer.is_alive() and _resource_initializer.ident is None:
            _resource_initializer.start()
        return _resource_initializer


def wait_for_resource_initialization() -> None:
    with _initializer_lock:
        if _resource_initializer is None:
            logger.error(
                "Resource initializer is None - resources were not initialized."
            )
            exception = RuntimeError(
                "Resource initializer is None - resources were not initialized."
            )
            attach_custom_error_code(
                exception, ErrorCodes.RESOURCE_INITIALIZATION_FAILED
            )
            raise exception
        else:
            _resource_initializer.join(timeout=300)  # wait at most 300 seconds

    if _resource_initializer is not None and _resource_initializer.is_alive():
        logger.error(
            "Resource initialization failed - initializer thread has been running for over 300 seconds."
        )
        exception = RuntimeError(
            "Resource initialization failed - initializer thread has been running for over 300 seconds."
        )
        attach_custom_error_code(exception, ErrorCodes.RESOURCE_INITIALIZATION_FAILED)
        raise exception


def set_upload_jars(upload: bool) -> None:
    """Set whether to upload jars required for Scala UDFs. This should be set to False if Scala UDFs
    are not used, to avoid the overhead of uploading jars."""
    global _upload_jars
    _upload_jars = upload
