import os
import subprocess
from pathlib import Path
from typing import Dict, List, Optional

import yaml
from aext_environments_server.conda_commands import CondaCommandsWrapper
from aext_environments_server.consts import CONDA_KERNEL_SPEC_PREFIX, EnvironmentStatus
from aext_environments_server.services.static_files import envs_static_file_service
from aext_environments_server.utils import extract_packages_from_env_file

from aext_shared import logger as custom_logger

logger = custom_logger.logger


class EnvironmentsService:

    def filename(self, env_name: str) -> str:
        return f"{env_name}.yml"

    def _save_environment_file(self, env_name: str, file_content: bytes):
        with open(self.filename(env_name), "wb") as f:
            f.write(file_content)

    def remove_environment_file(self, filename: str):
        os.remove(filename)

    async def list_quick_environments(self) -> List:
        static_environments = []
        listed_environments = CondaCommandsWrapper.list_environments()
        environments_index: Dict = await envs_static_file_service.get_index_file()

        if environments_index:
            env_files = await envs_static_file_service.get_all_env_files(environments_index)
            for env_name in env_files.keys():
                env_data: Dict = environments_index[env_name]["metadata"]
                env_data["all_packages"] = extract_packages_from_env_file(env_name, env_files[env_name])
                is_installed = CondaCommandsWrapper.is_installed(listed_environments, env_name)
                env_data["status"] = EnvironmentStatus.INSTALLED if is_installed else EnvironmentStatus.NOT_INSTALLED
                env_data["kernelspec"] = f"{CONDA_KERNEL_SPEC_PREFIX}{env_name}"
                static_environments.append(env_data)

        return static_environments

    async def download_environment_file(self, env_name: str):
        env_files = await envs_static_file_service.get_all_env_files()
        env_definition: bytes = env_files[env_name]
        self._save_environment_file(env_name, env_definition)

    async def get_env_file_path(self, env_name: str) -> Path:
        file_path = Path(self.filename(env_name))
        if not file_path.exists():
            await self.download_environment_file(env_name)
        return file_path.resolve()

    async def install_environment(self, env_name: str, file_path: Optional[str] = None) -> Optional[subprocess.Popen]:
        if file_path is None:
            # If the file path is not provided, then we assume the user is installing one of the quick environments
            file_path = await self.get_env_file_path(env_name)

        return await CondaCommandsWrapper.install_environment(env_name, str(file_path))

    async def clean_up_installation(self, env_name: str):
        file_path = await self.get_env_file_path(env_name)
        if file_path.exists():
            file_path.unlink()

    async def remove_environment(self, env_name: str) -> Optional[subprocess.Popen]:
        return await CondaCommandsWrapper.remove_environment(env_name)

    async def install_kernel(self, env_name: str) -> Optional[subprocess.Popen]:
        return await CondaCommandsWrapper.install_kernel(env_name)

    async def remove_kernel(self, env_name: str) -> Optional[subprocess.Popen]:
        return await CondaCommandsWrapper.remove_kernel(env_name)

    async def conda_clean(self) -> Optional[subprocess.Popen]:
        return await CondaCommandsWrapper.conda_clean()

    async def is_installed(self, env_name: str) -> Optional[bool]:
        listed_environments = CondaCommandsWrapper.list_environments()
        return CondaCommandsWrapper.is_installed(listed_environments, env_name)

    async def yml_has_ipykernel(self, file_path: str) -> bool:
        """
        Check if the YML file has an ipykernel include in its dependencies.

        Args:
            file_path: Path to the YML file
        """
        with open(file_path, "r") as f:
            try:
                env = yaml.safe_load(f)
            except yaml.YAMLError as exc:
                logger.warning(f"Environment YAML file malformed {exc}")
                raise

            for dep in env.get("dependencies", []):
                if isinstance(dep, str):
                    if "ipykernel" in dep:
                        return True
                elif isinstance(dep, dict) and "pip" in dep:
                    for pip_dep in dep["pip"]:
                        if "ipykernel" in pip_dep:
                            return True
            return False


environment_service = EnvironmentsService()
