import asyncio
import json
from json import JSONDecodeError
from typing import Optional

import httpx
from httpx import HTTPError, NetworkError, TimeoutException

from aext_shared import logger as custom_logger
from aext_shared.config import SHARED_CONFIG

logger = custom_logger.logger


class IndexFileDownloadError(Exception):
    pass


class StaticFilesClient:
    def __init__(self):
        self.base_url = f"{SHARED_CONFIG['static_content']['url']}/shared/"
        self.max_retries = 3
        self.timeout = 2

    async def get_file(self, uri_file_path: str) -> bytes:
        url = f"{self.base_url}{uri_file_path}"
        last_exception = None

        for attempt in range(1, self.max_retries + 1):
            try:
                async with httpx.AsyncClient(timeout=self.timeout) as client:
                    response = await client.get(url)
                    response.raise_for_status()
                    return response.content
            except (TimeoutException, NetworkError, HTTPError, Exception) as ex:
                logger.info(f"Attempt {attempt} failed: {ex}")
                last_exception = ex
                await asyncio.sleep(0.8)

        logger.error(f"All {self.max_retries} attempts failed to fetch environment {url} {last_exception}")
        raise last_exception


class EnvironmentsStaticFilesService:
    def __init__(self, static_files_client: StaticFilesClient):
        self.client = static_files_client

    async def get_index_file(self, local_fallback: bool = False) -> dict:
        if local_fallback:
            raise NotImplementedError
        content = await self.client.get_file("anaconda-notebooks/environments/index.json")
        if not content:
            raise IndexFileDownloadError("Could not download environments index")
        try:
            return json.loads(content)
        except JSONDecodeError:
            return {}

    async def get_env_file(self, uri_file_path: str, local_fallback: bool = False) -> bytes:
        if local_fallback:
            raise NotImplementedError
        return await self.client.get_file(f"anaconda-notebooks/environments/{uri_file_path}")

    async def get_all_env_files(self, environments_index: Optional[dict] = None, local_fallback: bool = False) -> dict:
        if local_fallback:
            raise NotImplementedError

        if not environments_index:
            environments_index = await self.get_index_file()

        tasks = []
        for env_name in environments_index:
            env = environments_index[env_name]
            tasks.append(asyncio.tasks.create_task(self.get_env_file(env["environment_file_path"]), name=env_name))

        finished, pending = await asyncio.wait(tasks, timeout=10, return_when=asyncio.ALL_COMPLETED)
        successful_tasks = {}
        if pending:
            for task in pending:
                task.cancel()
        for task in finished:
            if exc := task.exception():
                logger.error(f"Failed to fetch environment {task.get_name()}: {exc}")
            else:
                successful_tasks[task.get_name()] = task.result()

        return successful_tasks


envs_static_file_service = EnvironmentsStaticFilesService(StaticFilesClient())
