import os
import shutil
from glob import glob
from typing import Dict, List

import aext_project_filebrowser_server as project_filebrowser
import tornado.web
from aext_project_filebrowser_server.consts import PROJECTS_FILES_DIR, FileType
from aext_project_filebrowser_server.handlers import ProjectListWebSocketRouteHandler
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from jupyter_server.auth.identity import IdentityProvider
from tornado.simple_httpclient import SimpleAsyncHTTPClient
from tornado.testing import AsyncHTTPTestCase
from tornado.web import Application

from .fixtures import setup_database, tear_down_database
from .test_assets.project_data import EMPTY_LOCAL_PROJECT, LOCAL_PROJECT


class SetupTestingDB(AsyncHTTPTestCase):
    """
    A class that sets up a test database for testing.
    """

    def create_project(self, project_data: Dict):
        project = LocalProject.create(project_data)
        project.save()
        return project

    def create_default_project(self):
        project = LocalProject.create(LOCAL_PROJECT)
        project.save()
        return project

    def create_empty_project(self):
        project = LocalProject.create(EMPTY_LOCAL_PROJECT)
        project.save()
        return project

    def create_project_with_sync_data(self, id_: str, name: str, contents: List):
        data = EMPTY_LOCAL_PROJECT
        data["id"] = id_
        data["name"] = name
        data["contents"] = contents

        project = LocalProject.create(data)
        project.save(self.db)
        return project

    def create_project_files(self, project: LocalProject):
        project_root_folder = f"{PROJECTS_FILES_DIR}/{project.id}"

        def _create_recursively(contents):
            for content in contents:
                if content.type == FileType.FILE:
                    open(f"{project_root_folder}/{content.key}", "w")
                elif content.type == FileType.DIRECTORY:
                    os.makedirs(f"{project_root_folder}/{content.key}", exist_ok=True)
                    _create_recursively(content.contents)

        _create_recursively(project.data.contents)


class BasicAppHTTPTests(AsyncHTTPTestCase):
    """
    A class that sets up a test Basic Tornado App using the extension routes.
    """

    def get_app(self):
        return tornado.web.Application(
            project_filebrowser.get_routes("/"),
            cookie_secret="abc",
            identity_provider=IdentityProvider(token="abc"),
        )


def clean_projects():
    for fn in glob(f"{PROJECTS_FILES_DIR}/*"):
        shutil.rmtree(fn)


class BasicDBAppHTTPTests(SetupTestingDB, BasicAppHTTPTests):
    """
    A class that sets up a test database and a Basic Tornado App using the extension routes.
    """

    def setUp(self):
        super(SetupTestingDB, self).setUp()
        print("Setting up test...")
        # tearDown might not be called (in case an exception happens during the test execution)
        # making sure it is cleaned before setting up a new one
        self.db = setup_database()

    def tearDown(self):
        super(SetupTestingDB, self).tearDown()
        print("Tearing down test...")
        tear_down_database()

    def read_file(self, file_path: str):
        with open(file_path, "r") as file:
            return file.read()

    def _request_data(self):
        headers = {"Authorization": "token abc"}
        return headers

    def create_project(self, project_data):
        project = LocalProject.create(project_data)
        project.save(self.db)
        return project


class WebSocketBaseTestCase(SetupTestingDB):
    """
    A class that sets up a test database.
    """

    def setUp(self):
        super(SetupTestingDB, self).setUp()
        print("Setting up test...")
        self.db = setup_database()

    def tearDown(self):
        super(SetupTestingDB, self).tearDown()
        print("Tearing down test...")
        tear_down_database()


class BasicAppWebSocketsTests(WebSocketBaseTestCase):
    def get_app(self):
        return Application(
            [
                ("/project/list", ProjectListWebSocketRouteHandler),
            ],
            cookie_secret="abc",
            identity_provider=IdentityProvider(token="abc"),
        )

    def get_http_client(self):
        # These tests require HTTP/1; force the use of SimpleAsyncHTTPClient.
        return SimpleAsyncHTTPClient()
