import copy
import json
import unittest
from pathlib import Path

from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProjectFileList,
    CloudProjectPermissions,
)
from aext_project_filebrowser_server.schemas.organizations import OrganizationMembers


class TestSchemas(unittest.TestCase):
    def setUp(self):
        with open(f"{Path.cwd()}/tests/test_assets/list_project_files_projects_response.json", "r") as f:
            self.list_files_response = json.loads(f.read())

    def test_cloud_project_file_list(self):
        self.assertTrue(CloudProjectFileList.create_from_cloud_response(self.list_files_response))

    def test_cloud_project_file_list_should_fail_if_info_is_missing_from_json(self):
        json_response = copy.deepcopy(self.list_files_response)
        del json_response["num_items"]
        with self.assertRaises(KeyError):
            CloudProjectFileList.create_from_cloud_response(json_response)

        json_response = copy.deepcopy(self.list_files_response)
        del json_response["items"][0]["name"]
        with self.assertRaises(TypeError):
            CloudProjectFileList.create_from_cloud_response(json_response)

    def test_cloud_project_permissions_create_from_cloud_response(self):
        permissions = CloudProjectPermissions.create_from_cloud_response(
            {
                "items": [
                    {"type": "user_id", "relation": "owner", "id": "e761da58-732e-4943-af40-5d4caba41bfe"},
                    {"type": "user_id", "relation": "writer", "id": "b761da58-732e-4943-af40-5d4caba41bfe"},
                    {"type": "user_id", "relation": "reader", "id": "b761da58-732e-4943-af40-5d4caba41bfe"},
                ],
                "num_items": 3,
            }
        )
        assert len(permissions.permissions) == 2
        assert permissions.get_object_permission("b761da58-732e-4943-af40-5d4caba41bfe").roles == ["writer", "reader"]

    def test_organization_members_create_from_cloud_response(self):
        org_members = OrganizationMembers.create_from_cloud_response(
            {
                "items": [
                    {
                        "id": "e761da58-732e-4943-af40-5d4caba41bfe",
                        "email": "user@anaconda.com",
                        "first_name": "Anaconda",
                        "last_name": "User",
                        "role": "admin",
                        "subscriptions": ["security_subscription"],
                        "groups": "null",
                    },
                    {
                        "id": "b51ba6a3-95c5-4895-b725-467d9f2f6013",
                        "email": "user+1@anaconda.com",
                        "first_name": "Anaconda +1",
                        "last_name": "User",
                        "role": "member",
                        "subscriptions": [],
                        "groups": "null",
                    },
                ],
                "total_count": 2,
                "filtered_count": 2,
            }
        )
        assert len(org_members.members) == 2
        assert org_members.get_member("b51ba6a3-95c5-4895-b725-467d9f2f6013").email == "user+1@anaconda.com"
