from typing import Any, Callable, Dict, List, Optional

from aext_project_filebrowser_server import logger as custom_logger
from aext_project_filebrowser_server.clients.base import ProjectAPIClientInterface
from aext_project_filebrowser_server.exceptions import (
    CloudProjectNameAlreadyExistsError,
    CloudProjectNotFoundError,
    CloudProjectUpdatePayloadError,
    ListUserPermissionsError,
    ProjectFailDelete,
    ProjectNameConflict,
    UserDoesNotBelongToOrganization,
)
from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProject,
    CloudProjectFileList,
    CloudProjectFileTree,
    CloudProjectPermissions,
    MyProjectPermissions,
    ProjectPermissionUpdate,
    UserProjectPermissions,
)
from aext_project_filebrowser_server.schemas.organizations import OrganizationMembers

from aext_shared.backend_proxy import ProxyResponse
from aext_shared.errors import BackendError

logger = custom_logger.logger


class ProjectAPIClientHTTP(ProjectAPIClientInterface):
    """
    A http-based implementation of ProjectAPIClientInterface.

    Methods are wrappers for the following endpoints: https://nucleus-latest.anacondaconnect.com/api/docs/projects.html.
    """

    @property
    def proxy(self) -> Callable:
        return self._proxy

    @proxy.setter
    def proxy(self, proxy: Callable) -> None:
        self._proxy = proxy

    async def fetch(self, path: str, **kwargs) -> ProxyResponse:
        return await self.proxy(path, **kwargs)

    async def get_projects(self, **kwargs) -> List[CloudProject]:
        projects = []

        response = await self.fetch("projects", **kwargs)
        if response["remote_status_code"] == 200:
            projects = response["remote_data"].get("items", [])

        return [CloudProject.create(project) for project in projects]

    async def get_project(self, project_id: str, **kwargs) -> Optional[CloudProject]:
        try:
            project = None
            # aext_shared.aext_shared.backend_proxy_core raises exceptions for the following conditions:
            # 1. response.status_code < httpx.codes.OK; or
            # 2. response.status_code >= httpx.codes.MULTIPLE_CHOICES
            response = await self.fetch(f"projects/{project_id}", **kwargs)
            if response["remote_status_code"] == 200:
                project = CloudProject.create(response["remote_data"])
            return project
        except BackendError as ex:
            if ex.data["remote_status_code"] == 403:
                raise CloudProjectNotFoundError from ex
            raise

    async def get_project_filetree(self, project_id: str, **kwargs) -> Optional[CloudProjectFileTree]:
        try:
            filetree = None
            # aext_shared.aext_shared.backend_proxy_core raises exceptions for the following conditions:
            # 1. response.status_code < httpx.codes.OK; or
            # 2. response.status_code >= httpx.codes.MULTIPLE_CHOICES
            response = await self.fetch(f"projects/{project_id}/file-tree", **kwargs)
            if response["remote_status_code"] == 200:
                filetree = CloudProjectFileTree(**response["remote_data"])
            return filetree
        except BackendError as ex:
            if ex.data["remote_status_code"] == 403:
                raise CloudProjectNotFoundError from ex
            raise

    async def list_project_files(self, project_id: str, **kwargs) -> Optional[CloudProjectFileList]:
        """
        Retrieves the list of files belonging to the project
        Args:
            project_id: id of the project
            **kwargs:

        Returns: Optionally returns the list of files

        """
        file_list = None

        response = await self.fetch(f"projects/{project_id}/files", **kwargs)
        if response["remote_status_code"] == 200:
            file_list = CloudProjectFileList.create_from_cloud_response(response["remote_data"])

        return file_list

    async def download_file(self, project_id: str, filename: str, **kwargs) -> Optional[bytes]:
        response = await self.fetch(f"projects/{project_id}/files/{filename}", **kwargs)
        if response.status_code == 200:
            return response.text

        logger.error(f"Failed to download item. Response: {response}")
        return None

    async def create_project(
        self, project_title: str, metadata: Optional[Dict] = None, **kwargs
    ) -> Optional[CloudProject]:
        """
        Creates a new project first hitting the Projects API and then returns a Project instance.

        Args:
            project_title: project's title

        Returns: a Project instance if project is successfully created in Project API
        """
        project = None

        payload = {"title": project_title, "metadata": metadata}
        try:
            response = await self.fetch("projects", method="POST", json=payload, **kwargs)
            status_code = response["remote_status_code"]
            if status_code == 201:
                project = CloudProject.create(response["remote_data"])
            return project
        except BackendError as ex:
            if ex.data["remote_status_code"] == 409:
                #  A name conflict was spotted - names are unique and if not provided are created out of the title
                raise ProjectNameConflict
            raise ex

    async def delete_project(self, project_id: str, **kwargs) -> bool:
        try:
            # aext_shared.aext_shared.backend_proxy_core raises exceptions for the following conditions:
            # 1. response.status_code < httpx.codes.OK; or
            # 2. response.status_code >= httpx.codes.MULTIPLE_CHOICES
            response = await self.fetch(f"projects/{project_id}", method="DELETE", **kwargs)
            status_code = response["remote_status_code"]
            if status_code == 204:
                return True
            return False
        except BackendError as ex:
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                # The Projects API returns a 403 (ideally a 404 would be returned)
                # when the project does not exist and that should
                # be a valid scenario where the extension proceeds with the local deletion
                return True
            msg = "Unexpected error while deleting project on cloud"
            logger.warning(msg)
            raise ProjectFailDelete(msg) from ex

    async def update_project(self, project_id: str, payload: Any, **kwargs) -> CloudProject:
        try:
            response = await self.fetch(f"projects/{project_id}", method="PATCH", json=payload, **kwargs)
            return CloudProject.create(response["remote_data"])
        except BackendError as ex:
            status_code = ex.data["remote_status_code"]
            error_message = ex.data["remote_data"]["error"]["message"]
            if status_code == 403:
                raise CloudProjectNotFoundError from ex
            elif status_code == 422:
                raise CloudProjectUpdatePayloadError from ex
            # TODO: handle the exception and set the error code directly on the Projects API
            elif status_code == 500 and "asyncpg.exceptions.UniqueViolationError" in error_message:
                raise CloudProjectNameAlreadyExistsError from ex
            raise

    async def list_organization_users(self, organization_name: str) -> OrganizationMembers:
        """
        List all members of an organization
        Args:
            organization_name: organization id

        Returns: OrganizationMembers object

        """
        try:
            response = await self.fetch(f"organizations/{organization_name}/users", method="GET")
            return OrganizationMembers.create_from_cloud_response(response["remote_data"])
        except BackendError as ex:
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                raise PermissionError from ex
            elif status_code == 404:
                raise CloudProjectNotFoundError from ex
            raise

    async def list_project_permissions(self, project_id: str) -> CloudProjectPermissions:
        """
        List all the permissions associated with a project
        Args:
            project_id: project's id

        Returns: CLoudProjectPermission object

        """
        try:
            response = await self.fetch(f"projects/{project_id}/permissions", method="GET")
            return CloudProjectPermissions.create_from_cloud_response(response["remote_data"])
        except BackendError as ex:
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                raise PermissionError from ex
            elif status_code == 404:
                raise CloudProjectNotFoundError from ex
            raise

    async def update_project_permissions(self, project_id: str, permissions: List[ProjectPermissionUpdate]) -> bool:
        """
        Updates project's permission for a given organization
        Args:
        project_id: project's id
        """
        try:

            payload = {"permissions": [permission.create_payload() for permission in permissions]}
            response = await self.fetch(
                f"projects/{project_id}/permissions",
                json=payload,
                method="POST",
                headers={"app": "cloud-notebooks-filebrowser"},
            )
            if response["remote_status_code"] > 199 and response["remote_status_code"] < 300:
                return True
            return False
        except BackendError as ex:
            logger.exception(ex)
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                raise PermissionError from ex
            elif status_code == 404:
                raise CloudProjectNotFoundError from ex
            raise

    async def update_file_metadata(
        self, project_id: str, filename: str, metadata: Dict, description: Optional[str] = ""
    ) -> bool:
        """
        Updates file's metadata
        Args:
            project_id: id of the project that the file belongs to
            filename: file's filename (key)
            metadata: dict containing all the metadata
            description: file description

        Returns: bool

        Raises: PermissionError, CloudProjectNotFoundError
        """
        payload = {"metadata": metadata}

        if description:
            payload["description"] = description
        try:
            response = await self.fetch(f"projects/{project_id}/files/{filename}", json=payload, method="PATCH")
            if response["remote_status_code"] > 199 and response["remote_status_code"] < 300:
                return True
            return False
        except BackendError as ex:
            logger.exception(ex)
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                raise PermissionError from ex
            elif status_code == 404:
                raise CloudProjectNotFoundError from ex
            raise

    async def list_user_permissions(self, project_id: str) -> UserProjectPermissions:
        """
        List all the permissions from all users related to a project. Only owners can fetch this info.
        Args:
            project_id: id of the project that the file belongs to

        Returns: List

        Raises: PermissionError
        """

        try:
            response = await self.fetch(f"projects/{project_id}/permissions", method="GET")
            if 199 < response["remote_status_code"] < 300:
                return UserProjectPermissions(**response["remote_data"])
            raise ListUserPermissionsError
        except BackendError as ex:
            logger.exception(ex)
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                raise PermissionError from ex
            elif status_code == 404:
                raise CloudProjectNotFoundError from ex
            raise

    async def list_my_permissions(self, project_id: str) -> MyProjectPermissions:
        """
        List user's permissions for a particular project

        Return: MyProjectPermissions

        Raises: PermissionError
        """
        try:
            response = await self.fetch(f"projects/{project_id}/permissions/my", method="GET")
            if 199 < response["remote_status_code"] < 300:
                return MyProjectPermissions.create(response["remote_data"])
            raise ListUserPermissionsError
        except BackendError as ex:
            logger.exception(ex)
            status_code = ex.data["remote_status_code"]
            if status_code == 403:
                raise PermissionError from ex
            elif status_code == 404:
                raise CloudProjectNotFoundError from ex
            raise
