import os
import platform
from enum import Enum

from aext_shared.config import ConfigValueFactory, Environment, get_environment_v2


class OSPlatformTypes(str, Enum):
    WINDOWS = "WINDOWS"
    LINUX = "LINUX"
    MACOS = "DARWIN"


ENVIRONMENT = get_environment_v2(ConfigValueFactory())
OS_PLATFORM = platform.system().upper()
IS_LOCAL_ENVIRONMENT = Environment.is_local(ENVIRONMENT)
IS_CLOUD_ENVIRONMENT = Environment.is_cloud(ENVIRONMENT)
IS_WINDOWS = OSPlatformTypes.WINDOWS == OS_PLATFORM
IS_LINUX = OSPlatformTypes.LINUX == OS_PLATFORM
IS_MACOS = OSPlatformTypes.MACOS == OS_PLATFORM
ANACONDA_PROJECTS_FOLDER = "anaconda_projects"

FILEBROWSER_LOCAL_ENVIRONMENT_BASE_DIR = os.path.join(os.getcwd(), ANACONDA_PROJECTS_FOLDER)
FILEBROWSER_CLOUD_ENVIRONMENT_BASE_DIR = os.path.join("/var", "www", "filebrowser")
FILEBROWSER_PROJECT_URL_RELATIVE = "/aext_project_filebrowser_server/project"

# The Symlink is only created in the cloud environment
# where we store the file in a different location (FILEBROWSER_CLOUD_ENVIRONMENT_BASE_DIR)
# The symlink is necessary because the files are accessed via jupyter content's manager
# which can only access files within the user's Jupyter home folder, therefore the link
# is created so the content manager can access the files
TARGET_PROJECTS_SYMLINK = os.path.join(os.path.expanduser("~"), ANACONDA_PROJECTS_FOLDER)


def get_projects_default_base_dir() -> str:
    """
    Returns the default BASE path for storing project files.
    """
    if IS_LOCAL_ENVIRONMENT:
        return FILEBROWSER_LOCAL_ENVIRONMENT_BASE_DIR
    return FILEBROWSER_CLOUD_ENVIRONMENT_BASE_DIR


def get_projects_file_dir():
    """
    Returns the path where the projects folder are stored
    """
    if IS_LOCAL_ENVIRONMENT:
        return ""  # for local environments it is saved straight to the `anaconda_projects` folder
    return ".projects"


# JupyterLab gets the user dir using the exactly same way.
# Reference: https://github.com/jupyter-server/jupyter_server/blob/main/jupyter_server/serverapp.py#L399
USER_JLAB_HOME_PATH = os.getenv("USER_JLAB_HOME_PATH", os.path.expanduser("~"))
PROJECTS_BASE_DIR = os.getenv("PROJECTS_BASE_DIR", get_projects_default_base_dir())
PROJECTS_FILES_DIR = os.path.join(PROJECTS_BASE_DIR, get_projects_file_dir())
PROJECTS_API_VERSION = "2023.11.28"

UPLOAD_FILE_LOG_FILE = os.path.join("/tmp", "file-upload-metrics.log")
UNDEFINED_ENVIRONMENT = "undefined_environment"

FILE_MAX_SIZE_100_MB = 100 * 1024 * 1024  # 100 MB
DEFAULT_PROJECT_TAGS = ["data", "notebook"]
CODE_SNIPPET_TAG = "code_snippet"
VALID_PROJECT_TAGS = DEFAULT_PROJECT_TAGS + [CODE_SNIPPET_TAG]
FIVE_MINUTES_TIMEOUT = 60 * 5
THIRTY_SECONDS_TIMEOUT = 30

ALEMBIC_DIR = os.path.join(os.path.dirname(os.path.abspath(__file__)), "alembic")


class FileType(str, Enum):
    FILE = "file"
    DIRECTORY = "directory"
    PROJECT = "project"


class LocalStorageSyncState(str, Enum):
    # order defines status priority
    SYNC_ERROR = "sync_error"
    SYNCING = "syncing"
    HAS_CONFLICT = "has_conflict"
    UNSTAGED = "unstaged"
    SYNCED = "synced"

    @classmethod
    def list_values(cls):
        return [element.value for element in cls]


class MetricNames(str, Enum):
    UPLOAD_ELAPSED_TIME = "FILE_UPLOAD_ELAPSED_TIME"


class FileOperations(str, Enum):
    LOCAL_DELETE = "local_delete"
    LOCAL_WRITE = "local_write"
    CLOUD_DELETE = "cloud_delete"
    CLOUD_WRITE = "cloud_write"


class RequestMethods(str, Enum):
    GET = "GET"
    PUT = "PUT"
    PATCH = "PATCH"
    POST = "POST"
    DELETE = "DELETE"


class ProjectPermissionRoles(str, Enum):
    OWNER = "owner"
    WRITER = "writer"
    READER = "reader"

    @classmethod
    def list_values(cls):
        return [element.value for element in cls]


class ProjectPermissionAction(str, Enum):
    ADD = "add"
    REMOVE = "remove"

    @classmethod
    def list_values(cls):
        return [element.value for element in cls]


class ProjectPermissionActionType(str, Enum):
    USER = "user_id"
    ORGANIZATION = "org_id"

    @classmethod
    def list_values(cls):
        return [element.value for element in cls]


class SnippetLanguage(str, Enum):
    PYTHON = "python"


class WebsocketMessages(str, Enum):
    START = "start"
    PYTHON_SNIPPETS = "python_snippets"


class InternalOrigins(str, Enum):
    WATCHDOG = "aext_project_filebrowser_server/services/watchdog"
