import os
from typing import Dict

import httpx
from aext_project_filebrowser_server import logger as custom_logger
from aext_project_filebrowser_server.request_utils import cloud_request
from aext_project_filebrowser_server.utils import get_boolean_env_var

logger = custom_logger.logger


class SnakeEyes:
    """Snake eyes client - Sends metrics/logs to Anaconda Snake Eyes"""

    EVENT_PREFIX = "filebrowser"
    EVENT_CREATE_PROJECT = f"{EVENT_PREFIX}_create_project"
    EVENT_DELETE_PROJECT = f"{EVENT_PREFIX}_delete_project"
    EVENT_UPDATE_PROJECT = f"{EVENT_PREFIX}_update_project"
    EVENT_PULL_ALL = f"{EVENT_PREFIX}_pull_all"
    EVENT_PULL_ONE = f"{EVENT_PREFIX}_pull_one"
    EVENT_PUSH_ALL = f"{EVENT_PREFIX}_push_all"
    EVENT_PUSH_ONE = f"{EVENT_PREFIX}_push_one"
    EVENT_CREATE_FILE = f"{EVENT_PREFIX}_create_file"
    EVENT_ADD_FILE = f"{EVENT_PREFIX}_add_file"
    EVENT_UPLOAD_FILE = f"{EVENT_PREFIX}_upload_file"
    EVENT_DELETE_FILE = f"{EVENT_PREFIX}_delete_file"
    EVENT_UPDATE_FILE = f"{EVENT_PREFIX}_update_file"
    EVENT_RENAME_FILE = f"{EVENT_PREFIX}_rename_file"
    EVENT_UPDATE_PROJECT_PERMISSIONS = f"{EVENT_PREFIX}_update_project_permissions"

    async def send(self, user_credentials: Dict, metric_data: Dict) -> bool:
        """
        Send metric data to Snake Eyes
        Args:
            user_credentials: user's Anaconda Cloud auth credentials
            metric_data: json containing all the relevant data

        Returns: boolean indicating success/failure

        """
        # return false for watchdog requests (no access token available)
        if not user_credentials.get("access_token", None):
            return False

        if not get_boolean_env_var("COLLECT_METRICS", default=True):
            logger.debug("Metrics are OFF")
            return False

        try:
            logger.debug(f"Sending metric: {metric_data}")
            response = await cloud_request(
                "snake-eyes/record",
                user_credentials=user_credentials,
                method="POST",
                timeout=3,
                json_payload=metric_data,
            )
            logger.debug(f"Metric response {response}")
            if 199 < response.status_code < 300:
                return True
            return False
        except httpx.TimeoutException:
            logger.error("Timeout while writing file snake-eyes metrics")
            return False
        except httpx.NetworkError:
            logger.error("Network error while sending snake-eyes metrics")
            return False
        except Exception:
            logger.error("Error while sending snake-eyes metrics")
            return False


metrics = SnakeEyes()
