import io
import os
import sqlite3
from pathlib import Path
from typing import Dict

import alembic.config
from aext_project_filebrowser_server import logger as custom_logger
from aext_project_filebrowser_server.consts import (
    ALEMBIC_DIR,
    ENVIRONMENT,
    OS_PLATFORM,
    PROJECTS_BASE_DIR,
    OSPlatformTypes,
)
from aext_project_filebrowser_server.exceptions import (
    AlembicConfigDBPropertyNotFound,
    AlembicConfigNotFound,
)
from alembic import command

from aext_shared.config import Environment

logger = custom_logger.logger


class AlembicConfig:
    SQLITE_URL_PREFIX = "sqlite:///"

    DB_FILENAME = "project_filebrowser.db"
    DB_PATH_KEY = "db_path"
    DB_CONN_URL_KEY = "db_conn_url"
    UNITTEST_LOCAL = "unittest_local"

    DB_UNITTEST_FILENAME = "project_filebrowser-unittests.db"

    def __init__(self, environment, os_platform, project_base_dir):
        self.ENVIRONMENT = environment
        self.OS_PLATFORM = os_platform
        self.DB_BASE_DIR = project_base_dir
        self.GH_RUNNER_BASE_DIR = "/tmp"

        self.DB_PATHS = {
            OSPlatformTypes.LINUX: {
                Environment.cloud_production: os.path.join(self.DB_BASE_DIR, self.DB_FILENAME),
                Environment.cloud_development: os.path.join(self.DB_BASE_DIR, self.DB_FILENAME),
                Environment.local_production: os.path.join(self.DB_BASE_DIR, "db", self.DB_FILENAME),
                Environment.local_development: os.path.join(self.DB_BASE_DIR, "db", self.DB_FILENAME),
                Environment.github_actions: os.path.join(self.GH_RUNNER_BASE_DIR, self.DB_UNITTEST_FILENAME),
            },
            OSPlatformTypes.MACOS: {
                Environment.cloud_production: os.path.join(self.DB_BASE_DIR, self.DB_FILENAME),
                Environment.cloud_development: os.path.join(self.DB_BASE_DIR, self.DB_FILENAME),
                Environment.local_production: os.path.join(self.DB_BASE_DIR, "db", self.DB_FILENAME),
                Environment.local_development: os.path.join(self.DB_BASE_DIR, "db", self.DB_FILENAME),
                Environment.github_actions: os.path.join(self.GH_RUNNER_BASE_DIR, self.DB_UNITTEST_FILENAME),
            },
            OSPlatformTypes.WINDOWS: {
                Environment.cloud_production: os.path.join(self.DB_BASE_DIR, self.DB_FILENAME),
                Environment.cloud_development: os.path.join(self.DB_BASE_DIR, self.DB_FILENAME),
                Environment.local_production: os.path.join(self.DB_BASE_DIR, "db", self.DB_FILENAME),
                Environment.local_development: os.path.join(self.DB_BASE_DIR, "db", self.DB_FILENAME),
                Environment.github_actions: os.path.join(self.GH_RUNNER_BASE_DIR, self.DB_UNITTEST_FILENAME),
            },
            self.UNITTEST_LOCAL: {
                Environment.local_development: os.path.join(self.DB_BASE_DIR, "db", self.DB_UNITTEST_FILENAME),
            },
        }
        try:
            self.DB_PATH = self.DB_PATHS[self.OS_PLATFORM][self.ENVIRONMENT]
            self.DB_CONNECTION_URL = f"{self.SQLITE_URL_PREFIX}{self.DB_PATHS[self.OS_PLATFORM][self.ENVIRONMENT]}"
        except KeyError as ex:
            logger.error(f"Alembic config not found {self.OS_PLATFORM} {self.ENVIRONMENT} - {ex}")
            raise AlembicConfigNotFound()

    def get_config_data(self) -> Dict:
        return {self.DB_CONN_URL_KEY: self.DB_CONNECTION_URL, self.DB_PATH_KEY: self.DB_PATH}

    def get_db_connection_url(self) -> str:
        return self.DB_CONNECTION_URL

    def get_db_path(self) -> str:
        config_data = self.get_config_data()
        return config_data[self.DB_PATH_KEY]

    def get_unittest_db_path(self) -> str:
        try:
            return self.DB_PATHS[self.UNITTEST_LOCAL][Environment.local_development]
        except KeyError:
            raise AlembicConfigDBPropertyNotFound()

    def get_db_connection_unittest_url(self) -> str:
        try:
            return f"{self.SQLITE_URL_PREFIX}{self.get_unittest_db_path()}"
        except KeyError:
            raise AlembicConfigDBPropertyNotFound()

    def get_gh_runner_db_path(self) -> str:
        try:
            return self.DB_PATHS[self.OS_PLATFORM][Environment.github_actions]
        except KeyError:
            raise AlembicConfigDBPropertyNotFound()

    def get_db_connection_gh_runner_url(self) -> str:
        try:
            return f"{self.SQLITE_URL_PREFIX}{self.get_gh_runner_db_path()}"
        except KeyError:
            raise AlembicConfigDBPropertyNotFound()


alembic_config = AlembicConfig(ENVIRONMENT, OS_PLATFORM, PROJECTS_BASE_DIR)


def init_database_file():
    def create_folder_if_not_exists(folder_path):
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)

    # Specify the path for the SQLite database file
    db_file = alembic_config.get_db_path()
    logger.info(f"Checking for database file @ {db_file}")

    create_folder_if_not_exists(os.path.dirname(db_file))
    if Path(db_file).exists():
        logger.info("Database file already exists")
        return

    # Create a connection to the SQLite database file
    logger.info("Creating database file...")
    conn = sqlite3.connect(db_file)

    # Close the connection
    conn.close()


def migration_upgrade():
    logger.info(f"--> Alembic Config: {alembic_config.get_config_data()}")
    logger.info(f"Default dir: {os.getcwd()}. Running chdir...")
    try:
        init_database_file()
        output_buffer = io.StringIO()
        alembic_cfg = alembic.config.Config()
        alembic_cfg.set_main_option("script_location", ALEMBIC_DIR)
        alembic_cfg.set_main_option("sqlalchemy.url", alembic_config.get_db_connection_url())

        command.upgrade(alembic_cfg, "head")
        logger.info("Migrations executed")
        return output_buffer.getvalue()
    except Exception as ex:
        logger.exception(f"Error running migration {ex}")
        raise


def migration_list():
    output_buffer = io.StringIO()
    alembic_cfg = alembic.config.Config()
    alembic_cfg.set_main_option("script_location", ALEMBIC_DIR)
    alembic_cfg.set_main_option("sqlalchemy.url", alembic_config.get_db_connection_url())
    try:
        try:
            command.history(alembic_cfg)
        except alembic.util.exc.CommandError:
            command.history("./alembic.ini")
    except Exception as ex:
        logger.error(ex)
        raise

    return output_buffer.getvalue()
