import asyncio
import json
import multiprocessing
import os
import re
import shutil
import stat
from dataclasses import dataclass
from datetime import datetime
from typing import Any, Callable, Coroutine, Dict, List, Optional, Tuple, Union
from urllib.parse import parse_qs, urlencode, urlparse, urlunparse

import httpx
from aext_project_filebrowser_server import logger as custom_logger
from aext_project_filebrowser_server.consts import (
    FILE_MAX_SIZE_100_MB,
    PROJECTS_FILES_DIR,
    FileType,
    LocalStorageSyncState,
    RequestMethods,
)
from aext_project_filebrowser_server.request_utils import cloud_request
from jupyter_client.kernelspec import KernelSpecManager

from aext_shared.config import ConfigValueFactory, Environment, get_environment_v2

logger = custom_logger.logger
config_factory = ConfigValueFactory()


def get_mock(path: Union[str, os.PathLike]) -> str:
    with open(path, encoding="utf-8") as mock:
        return mock.read()


def is_hidden_folder_or_file(filepath):
    """Check if a file or directory is hidden."""
    if os.name == "nt":  # Windows
        attribute = os.stat(filepath).st_file_attributes
        return bool(attribute & stat.FILE_ATTRIBUTE_HIDDEN)
    else:  # Unix-like system
        return os.path.basename(filepath).startswith(".")


def is_navigator() -> bool:
    return get_environment_v2(config_factory) == Environment.local_production


def is_local_development() -> bool:
    return get_environment_v2(config_factory) == Environment.local_development


def is_github_actions_runner() -> Union[str, None]:
    return os.getenv("GITHUB_ACTIONS")


def is_pytest_running() -> Union[str, None]:
    return os.getenv("PYTEST_RUNNING")


def sphinx_running() -> Union[str, None]:
    return os.getenv("SPHINX_RUNNING")


async def run_concurrently(coroutines: Dict[str, Coroutine], timeout: Optional[int] = None) -> Dict[str, Any]:
    """
    Receives a list of coroutines and execute them all concurrently
    returning a dictionary where each key carries the coroutine name and it"s
    result as value.

    Args:
        coroutines: dict of coroutines keyed by some unique ID
        timeout: after this timeout the coroutines that have finished will be returned and the
                 remaining ones will keep in pending state

    Returns: Dict with coroutine IDs and it"s results

    """
    if not coroutines:
        return {}

    tasks = []
    for key in coroutines:
        tasks.append(asyncio.tasks.create_task(coroutines[key], name=key))

    finished, pending = await asyncio.wait(tasks, timeout=timeout, return_when=asyncio.ALL_COMPLETED)
    if pending:
        for task in pending:
            task.cancel()

    return {task.get_name(): task.result() for task in finished}


async def run_coroutine(
    futures: List["SyncCoroutine"], result_processor: Optional[Callable] = None  # noqa
) -> Optional[List[Dict]]:
    """
    Runs a set of coroutines concurrently.

    Args:
        futures: A list of SyncCoroutine instances
        result_processor: An optional callable that will be called to process the results

    Returns: A list containing booleans that reflects on whether the coroutines executed successfully
    """
    coroutines = {ft.function_id: ft.function(*ft.function_args) for ft in futures}
    if coroutines:
        result = await run_concurrently(coroutines)
        if result_processor:
            try:
                result_processor(result)
            except Exception as ex:
                logger.error(f"Failed to process the sync results {ex}")
        return result


def run_process(target: Callable, args: Tuple = ()) -> multiprocessing.Process:
    """
    Creates a new process using multiprocessing lib and starts it.
    Args:
        target: function that will be executed in the process
        args: function arguments

    Returns: the process instance
    """
    # XXX this is triggering the migrations because it is now available in __init__
    process = multiprocessing.Process(target=target, args=args)
    process.start()
    logger.info(f"Starting Process {process.pid}: {target}, {args}")
    return process


def get_username() -> str:
    """
    Gets username account
    """
    return os.getenv("JUPYTERHUB_USER", os.getenv("USERNAME", os.getenv("USER", "unknown")))


def file_exists(file_path: [str, os.PathLike]) -> bool:
    return os.path.exists(file_path)


def is_file(file_path: Union[str, os.PathLike]) -> bool:
    return os.path.isfile(file_path)


def is_dir(path: Union[str, os.PathLike]) -> bool:
    return os.path.isdir(path)


def is_folder_empty(path: Union[str, os.PathLike]) -> bool:
    return len(os.listdir(path)) == 0


def create_empty_file(file_path: Union[str, os.PathLike]) -> None:
    with open(file_path, "wb"):
        pass


def copy_file(source: Union[str, os.PathLike], destination: Union[str, os.PathLike]) -> None:
    shutil.copyfile(source, destination)

    if not is_file(destination):
        message = f"Error copying file from {source} to {destination}"
        raise Exception(message)


def read_file_content(file_path: Union[str, os.PathLike]) -> bytes:
    with open(file_path, "rb") as f:
        return f.read()


def get_file_full_path(file_relative_path: str, project_id: str) -> str:
    return os.path.join(PROJECTS_FILES_DIR, project_id, file_relative_path)


def get_file_size(file_path: Union[str, os.PathLike]) -> int:
    try:
        file_content = read_file_content(file_path)
        return len(file_content)
    except Exception:
        return None


def write_file_content(file_path: Union[str, os.PathLike], content: bytes) -> bytes:
    with open(file_path, "w") as f:
        return f.write(content)


def delete_file(path: Union[str, os.PathLike]) -> bool:
    if is_file(path):
        os.remove(path)
        return True
    return False


def delete_folder(path: Union[str, os.PathLike]) -> bool:
    if is_dir:
        shutil.rmtree(path, ignore_errors=True)
        return True
    return False


def get_kernel_specs() -> Dict:
    """
    List all kernel specs
    Returns: A dict containing the kernel specs

    """
    ksm = KernelSpecManager()
    kernel_specs = ksm.get_all_specs()
    return kernel_specs


def list_kernel_specs_names() -> List:
    """
    List all the kernel spec names available
    Returns: List containing the kernel names

    """
    kernel_specs = get_kernel_specs()
    return list(kernel_specs.keys())


def get_latest_kernel_spec_name() -> Union[str, None]:
    """
    Returns the latest kernel spec name
    Returns: the latest kernel name

    """
    try:
        return list_kernel_specs_names()[0]
    except IndexError:
        return None


def sanitize_directory_path(dir_name: str = None) -> str:
    """
    Sanitizes a directory path by removing unnecessary elements, such as leading/trailing slashes,
    relative paths and redundant separators.

    If no directory name is provided or if it's the current working directory ("." or "./"),
    an empty string is returned. Otherwise, the function strips leading/trailing whitespace
    and slashes from the input path, then iteratively removes leading relative paths.
    It also normalizes the path by removing redundant separators, removes any remaining relative
    path elements, and reconstructs the sanitized path using the OS-specific separator.
    If the original directory name ends with a slash, the function adds it to the sanitized path.

    Args:
        dir_name: The directory path to sanitize (default is None).

    Returns:
        A sanitized directory path as a string.
    """

    root_filter = [".", "./"]
    if not dir_name or dir_name in root_filter:
        return ""

    # Remove leading/trailing whitespace and slashes
    dir_name = dir_name.strip().strip("/\\")

    # Remove leading relative paths (../ or ..\)
    while dir_name.startswith(".."):
        dir_name = dir_name[2:].strip("/\\")

    # Remove redundant slashes
    dir_name = os.path.normpath(dir_name)

    # Remove any remaining relative path elements
    parts = dir_name.split(os.sep)
    sanitized_parts = [part for part in parts if part not in ("..", ".")]

    # Reconstruct the sanitized path
    sanitized_path = os.sep.join(sanitized_parts)

    # Ensure the path ends with a slash if the original did

    # Check if the original dir_name ends with a slash
    ends_with_slash = sanitized_path.endswith("/") or sanitized_path.endswith("\\")
    if not ends_with_slash:
        sanitized_path += os.sep

    return sanitized_path


def sanitize_filename_path(filename_path: str = None, keep_leading_slash: bool = False) -> str:
    """
    Sanitizes a filename path by removing unnecessary elements, such as leading/trailing slashes,
    relative paths and redundant separators.

    Args:
        filename_path: The filename path to sanitize (default is None).

    Returns:
        A sanitized filename path as a string.
    """
    sanitized_file_path = sanitize_directory_path(filename_path)

    # keep the leading slash
    if keep_leading_slash:
        sanitized_file_path = os.sep + sanitized_file_path

    # Check if the original filename ends with a slash and remove it
    ends_with_slash = sanitized_file_path.endswith("/") or sanitized_file_path.endswith("\\")
    if ends_with_slash:
        sanitized_file_path = sanitized_file_path[:-1]

    return sanitized_file_path


def write_elapsed_time(metric_name: str, elapsed_time: float, log_file_path: str, subject: str):
    with open(log_file_path, "a") as log:
        log.write(f"{datetime.now()} [{metric_name}][{subject}] - Elapsed time {elapsed_time} \n")


class FileValidation:
    def __init__(
        self,
        filename: Optional[str] = None,
        content: Optional[bytes] = None,
        content_type: Optional[str] = None,
        file_full_path: Optional[str] = None,
    ):
        self.filename = filename
        self.content = content
        self.file_full_path = file_full_path
        self.content_type = content_type
        if not file_full_path and not content:
            raise ValueError("Either file content or file full path should be provided")

    def validate_size(self, max_size_bytes: int = FILE_MAX_SIZE_100_MB) -> bool:
        """
        Checks if file is smaller than the limit
        Args:
            max_size_bytes: value that should be considered as max size

        Returns: if file is smaller or not (boolean)

        """
        if self.content:
            file_size = len(self.content)
        else:
            file_size = get_file_size(self.file_full_path)

        return file_size <= max_size_bytes

    def is_valid(self, max_size_bytes: int = FILE_MAX_SIZE_100_MB) -> bool:
        """
        Validator function that makes sure to call every single validation
        that composes the file validation
        Returns: boolean telling whether the file is valid accordingly to all validations

        """
        is_size_valid = self.validate_size(max_size_bytes)
        return all([is_size_valid])


def get_intermediate_directories(file_path) -> List[Tuple[str, str]]:
    """
    Given a relative file path, it returns a list containing the path plus all the intermediates
    Args:
        file_path: a relative path to a file

    Returns: a list containing tuples where each tuple contains the file path and the file type

    """
    directories = [(file_path, FileType.FILE)]
    while file_path:
        file_path, tail = os.path.split(file_path)
        if tail:
            directories.append((file_path, FileType.DIRECTORY))
        else:
            if file_path:  # To include the root directory in non-Windows paths
                directories.append((file_path, FileType.DIRECTORY))
            break
    return [(_dir, _type) for _dir, _type in directories if _dir][::-1]


def generate_normalized_name(title: str) -> str:
    # Make lowercase
    name = title.lower()

    # Replace everything not [a-z0-9-_] with a space
    name = re.sub(r"[^a-z0-9-_]", " ", name)

    # Strip any leading or trailing whitespace
    name = name.strip()

    # Replace any series of spaces with a dash
    name = re.sub(r"\s+", "-", name)
    return name


def add_query_params(url: str, params: Dict):
    # Parse the URL into components
    url_parts = urlparse(url)

    # Convert the query string into a dictionary
    query_params = parse_qs(url_parts.query)

    # Update the dictionary with new parameters
    query_params.update(params)

    # Convert the dictionary back into a query string
    new_query_string = urlencode(query_params, doseq=True)

    # Reconstruct the URL with the new query string
    new_url_parts = url_parts._replace(query=new_query_string)
    new_url = urlunparse(new_url_parts)

    return new_url


def get_boolean_env_var(var_name, default=False):
    """
    Read an environment variable and convert it to a boolean.

    Args:
        var_name (str): The name of the environment variable.
        default (bool): The default value if the environment variable is not set.

    Returns:
        bool: The boolean value of the environment variable.
    """
    value = os.getenv(var_name, str(default))
    return value.lower() in ("true", "1", "t", "y", "yes")


async def evaluate_feature_flag(flag: str, user_credentials: Dict, default=False) -> bool:
    """
    Evaluates a feature flag and returns the result of the evaluation.

    Args:
        flag (str): The feature flag key;
        user_credentials (Dict): An object containing the user's credentials;
        default (bool): Default value to return if an exception occurs.

    Returns:
        A boolean representing the feature flag's evaluation.
    """
    try:
        flag_url = f"feature-flags/evaluate/notebooks/{flag}"
        flag_response = await cloud_request(flag_url, method=RequestMethods.POST, user_credentials=user_credentials)
        if 200 <= flag_response.status_code < 300:
            return True
        return False
    except Exception as e:
        logger.error(f"An unexpected error occurred while evaluating the feature flag {flag}: {e}")
        return default


def human_readable_size(size: int) -> str:
    """
    Convert a file size to a human-readable format.

    Args:
        size (int): The file size in bytes.

    Returns:
        str: A string representing the file size in a human-readable format (e.g., "123 Bytes", "456 KB", etc.).

    Notes:
        This function iterates through a list of units (Bytes, KB, MB, GB, TB, PB) and returns the first unit that can represent the given size. If the size is less than 1024 bytes, it will be displayed in its original form.
    """
    if size < 1:
        return "0 Bytes"

    for unit in ["Bytes", "KB", "MB", "GB", "TB", "PB"]:
        if size < 1024:
            return f"{size} {unit}"
        size //= 1024
    return f"{size} PB"


class DataclassEncoder(json.JSONEncoder):
    """
    A custom JSON encoder that specializes in serializing dataclasses and datetime objects.

    This encoder will serialize dataclasses by converting their instance dictionaries to JSON,
    and datetime objects by converting them to ISO format strings.
    For all other types of objects, it falls back to the default behavior of the underlying
    JSONEncoder class.

    Attributes:
        - dataclass: a boolean flag indicating whether the current object is a dataclass

    Methods:
        - default(obj): the main serialization method that determines how to serialize an object
    """

    def default(self, obj):
        if dataclass and hasattr(obj, "__dict__"):
            return obj.__dict__
        elif isinstance(obj, datetime):
            return obj.isoformat()
        else:
            return super().default(obj)


def get_max_priority(file_statuses: set(LocalStorageSyncState)) -> Optional[LocalStorageSyncState]:
    """
    Get the maximum priority folder status from a set of file statuses.

    This function takes a set of `LocalStorageSyncState` objects as input and returns
    the highest-priority status that is present in the set. If no folder status is found,
    it logs a warning message and returns None.

    Parameters:
        file_statuses (set): A set of `LocalStorageSyncState` objects representing the
        statuses of files or folders.

    Returns:
        str: The highest-priority folder status that is present in the input set,
        or None if no folder status is found.
    """
    status_priorities = LocalStorageSyncState.list_values()

    higher_status = 2**63 - 1  # positive integer infinity number
    for file_status in file_statuses:
        if status_priorities.index(file_status) < higher_status:
            higher_status = status_priorities.index(file_status)

    if higher_status == -1:
        logger.warning("Not possible to find folder status")
        return None

    folder_status = status_priorities[higher_status]
    return folder_status


def write_file_from_http_response(response: httpx.Response, file_path: str):
    content_type = response.headers.get("Content-Type", "").lower()
    if "text" in content_type or "json" in content_type or "xml" in content_type:
        # handle as a text file
        with open(file_path, "w", encoding="utf-8") as file:
            file.write(response.text)
        logger.info(f"Text file {file_path} saved.")
    else:
        # Handle as a binary file (e.g., image, PDF, etc.)
        with open(file_path, "wb") as file:
            file.write(response.content)
        logger.info("Binary file saved.")
