import os
import shutil
import uuid

from aext_project_filebrowser_server.consts import (
    IS_CLOUD_ENVIRONMENT,
    PROJECTS_BASE_DIR,
    PROJECTS_FILES_DIR,
)

from aext_shared import logger as custom_logger

logger = custom_logger.logger


def move_project_folders(src_dir=PROJECTS_BASE_DIR, dest_dir=PROJECTS_FILES_DIR):
    """
    Move project folders (folders named with a valid UUID) from one place to the other.
    The reason why this migration was originally created was due a bug which saves the projects
    in the wrong destination on CLOUD environments.

    The files were expected to be at /var/www/filebrowser/.projects but, instead, they were saved
    in /var/www/filebrowser.
    """
    os.makedirs(dest_dir, exist_ok=True)

    for item in os.listdir(src_dir):
        item_path = os.path.join(src_dir, item)

        if os.path.isdir(item_path):
            try:
                # checking UUID validity
                uuid.UUID(item, version=4)

                # Define destination folder path
                dest_path = os.path.join(dest_dir, item)

                # If the folder already exists the content will be overwritten
                if os.path.exists(dest_path):
                    shutil.rmtree(dest_path)
                    logger.debug(f"Overwritten existing folder: {dest_path}")

                # move to the destination
                shutil.move(item_path, dest_path)
                logger.debug(f"Moved folder: {item_path} to {dest_path}")

            except ValueError:
                logger.debug("Folder named with an invalid UUID4")
                continue

    logger.debug(f"Projects folder migrated from {src_dir} to {dest_dir}")


def run_projects_files_migration():
    logger.debug("Running Projects Files Migration")
    try:
        if IS_CLOUD_ENVIRONMENT:
            # Moving projects from /var/www/filebrowser/ to /var/www/filebrowser/.projects
            move_project_folders()
    except Exception as ex:
        logger.error(f"Failed to run Projects Files Migration {ex}")
