import os
import time
from typing import Awaitable, Callable, Dict, List, Optional, Tuple
from urllib.parse import quote

import httpx
from aext_project_filebrowser_server.consts import (
    FIVE_MINUTES_TIMEOUT,
    PROJECTS_FILES_DIR,
    UNDEFINED_ENVIRONMENT,
    UPLOAD_FILE_LOG_FILE,
    LocalStorageSyncState,
    MetricNames,
    RequestMethods,
)
from aext_project_filebrowser_server.exceptions import (
    CloudProjectNotFoundError,
    LocalProjectNotFoundError,
)
from aext_project_filebrowser_server.request_utils import (
    cloud_request,
    get_headers,
    make_request,
)
from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProject,
    CloudProjectFileTree,
)
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.schemas.projects import (
    SyncFileResult,
    UploadFileResult,
)
from aext_project_filebrowser_server.services.database import db
from aext_project_filebrowser_server.services.local_storage import local_storage
from aext_project_filebrowser_server.utils import (
    evaluate_feature_flag,
    get_username,
    is_file,
    read_file_content,
    run_concurrently,
    write_elapsed_time,
    write_file_from_http_response,
)

from aext_shared import logger as custom_logger
from aext_shared.backend_proxy import CERTIFICATES_LOCATION
from aext_shared.config import SHARED_CONFIG
from aext_shared.errors import BackendError

config = SHARED_CONFIG

logger = custom_logger.logger


class SyncCoroutine:
    """
    This class is only meant to be used as an argument for `run_coroutine`.
    There are 3 important attributes:

    function: reference to the function that will be called within run_coroutine, an awaitable
    function_args: a tuple holding all the arguments that will be passed as arguments to the sync_function
    function_id: a string used as a key for accessing the coroutine results later in `run_coroutine`
    """

    def __init__(self, function: Awaitable, function_args: Tuple, function_id: str):
        self.function = function
        self.function_args = function_args
        self.function_id = function_id


class ProjectSync:
    """
    An abstraction layer for all required project sync methods (pull and push).

    Notes:
        - expects injection of a ProjectService instance to communicate with the projects api;
        - imports the LocalStorage singleton used throughout the backend plugin.

    This class should be embedded directly inside ProjectFileBrowserHandler handlers.
    """

    @property
    def project_service(self) -> "ProjectService":  # noqa
        return self._project_service

    @project_service.setter
    def project_service(self, project_service: "ProjectService") -> None:  # noqa
        self._project_service = project_service

    async def pull_all(self) -> List[LocalProject]:
        """
        Implements an asynchronous pull mechanism for all projects.

        A pull can be summarised in the following steps:
            1. Uses the project service to retrieve all projects and their respective files from the cloud (async);
            2. Saves the projects' contents in the local database;
            3. Initiates the downloading of all files for all projects.

        Args: (None)

        Returns: a list containing each project's filetree (with statuses).
        """
        cloud_projects: List[CloudProject] = await self.project_service.get_cloud_projects()

        filetree_coroutines = {
            project.id: self.project_service.get_cloud_project_filetree(project.id) for project in cloud_projects
        }
        try:
            cloud_filetrees: Dict[str, CloudProjectFileTree] = await run_concurrently(filetree_coroutines)
        except BackendError as ex:
            logger.error(f"Error while fetching project file tree {ex}")
            raise

        local_projects = []
        for cloud_project in cloud_projects:
            if cloud_project and not cloud_project.validate_tags():
                continue

            if existing_local_project := LocalProject.get(cloud_project.id):
                updated_local_project = cloud_project.to_local_project(
                    cloud_filetrees.get(cloud_project.id).stored_files,
                    default_environment=existing_local_project.default_environment,
                    status=LocalStorageSyncState.SYNCING,
                )
                updated_local_project.update()
            else:
                updated_local_project = cloud_project.to_local_project(
                    cloud_filetrees.get(cloud_project.id).stored_files,
                    default_environment=UNDEFINED_ENVIRONMENT,
                    status=LocalStorageSyncState.SYNCING,
                )
                updated_local_project.save()

            await self.project_service.sync_pull_project_files(updated_local_project.id)
            local_projects.append(LocalProject.get(updated_local_project.id))

        # remove projects that have been deleted on the cloud but still remain in local db
        local_projects_db = await local_storage.list_projects()
        cloud_project_ids = [cloud_project.id for cloud_project in cloud_projects]
        for local_project in local_projects_db:
            if local_project.id not in cloud_project_ids:
                LocalProject.delete(local_project.id)
                await local_storage.delete_project_directory(local_project.id)

        return local_projects

    async def pull(self, project_id: str) -> Optional[LocalProject]:
        """
        Implements an asynchronous pull mechanism for a specific project.

        A pull can be summarised in the following steps:
            1. Uses the project service to retrieve all projects and their respective files from the cloud (async);
            2. Saves the projects' contents in the local database;
            3. Initiates the downloading of all files for all projects.

        Args:
            project_id: a string representing the project id.

        Returns: A project's filetree (with statuses).
        """

        coroutines = {
            "get_cloud_project": self.project_service.get_cloud_project(project_id),
            "get_cloud_project_filetree": self.project_service.get_cloud_project_filetree(project_id),
        }

        try:
            try:
                coroutine_results = await run_concurrently(coroutines, timeout=15)
            except BackendError as ex:
                logger.debug(f"Could not get project information from cloud {ex}")
                return None
            except Exception as ex:
                logger.error(f"Unexpected exception while pulling project {ex}")
                return None

            cloud_project: CloudProject = coroutine_results.get("get_cloud_project")
            if cloud_project and not cloud_project.validate_tags():
                return None

            cloud_file_tree: CloudProjectFileTree = coroutine_results.get("get_cloud_project_filetree")
        except CloudProjectNotFoundError:
            LocalProject.delete(project_id)
            await local_storage.delete_project_directory(project_id)
            return None

        # in case a project does not exist locally, a perfect copy of what we have
        # on the cloud will be saved into the database;
        # otherwise, the project already exists and its data shall be updated
        # NOTE: we should explicitly NOT update any file content in this block
        if existing_local_project := LocalProject.get(cloud_project.id):
            updated_local_project = cloud_project.to_local_project(
                cloud_file_tree.stored_files,
                default_environment=existing_local_project.default_environment,
                status=LocalStorageSyncState.SYNCING,
            )
            updated_local_project.update()
        else:
            updated_local_project = cloud_project.to_local_project(
                cloud_file_tree.stored_files,
                default_environment=UNDEFINED_ENVIRONMENT,
                status=LocalStorageSyncState.SYNCING,
            )
            updated_local_project.save()

        await self.project_service.sync_pull_project_files(project_id)
        return LocalProject.get(project_id)

    async def push(self, project_id: str, user_credentials: Dict) -> bool:
        """
        Pushes one specific project
        Args:
            project_id: id of the project that should be pushed

        Returns: boolean

        """
        if not LocalProject.get(project_id):
            raise LocalProjectNotFoundError

        return await self.project_service.sync_push_project_files(project_id)

    async def push_all(self) -> bool:
        """
        Pushes all users projects
        Returns: Dict containing the project id and if the process was spawned or not

        """
        projects = LocalProject.list()
        process_dispatch_status = {}
        for project in projects:
            was_process_created = await self.push(project.id)
            process_dispatch_status[project.id] = was_process_created
            if not was_process_created:
                logger.error(f"Could not create push process for {was_process_created}")

        return process_dispatch_status


async def sync_pull_delete_file(project_id: str, file_relative_path: str) -> SyncFileResult:
    """
    During a pull execution this function is called to delete a file from both
    local storage and database.
    Args:
        project_id: id that the file belongs to
        file_relative_path: file relative path

    Returns: boolean

    """
    try:
        logger.debug(f"Deleting... {file_relative_path}")
        project = LocalProject.get(project_id)
        # TODO: move this out of the coroutine - add it to sync_pull_result_processor
        project.remove_file_or_folder(file_relative_path, db_save=True)
        await local_storage.remove_object(project_id, file_relative_path)
        return SyncFileResult(True, project_id, file_relative_path, f"File {file_relative_path} deleted")
    except Exception as ex:
        logger.exception(f"Failed to remove file while pulling {ex}")
        return SyncFileResult(False, project_id, file_relative_path, f"Exception while pulling {ex}")


# INFO: The function is standalone because it needs to be pickled
async def sync_pull_write_file(
    project_id: str,
    file_relative_path: str,
    user_credentials: Dict,
    callback: Optional[Callable] = None,
    callback_args: Tuple = (),
    base_dir: str = PROJECTS_FILES_DIR,
    **kwargs,
) -> SyncFileResult:
    """
    This function is used to sync project_id's files from cloud (Projects service) to where this application is running (Cloud Notebooks, Navigator...).
    The function is currently executed in an external process using
    multiprocess Python's lib (see function run_process).

    Args:
        project_id: project_id's ID
        file_relative_path: is the key returned by Projects service - usually a path
        user_credentials: user's anaconda.cloud credentials - necessary to call the api on anaconda.cloud (after migration: anaconda.com)
        callback: in case it's necessary to run a function after the sync
        callback_args: args to be passed to the callback
        base_dir: project_id's base dir on the file system where the files should be placed
        **kwargs:

    Returns:

    """
    url = config["api"]["url"]
    url = f"{url}/projects/{project_id}/files/{file_relative_path}"
    logger.debug(f"Downloading... {url}")
    request = httpx.Request(
        "GET",
        url,
        headers=get_headers(user_credentials),
    )

    try:
        root_dir = os.path.join(base_dir, project_id)
        file_full_path = os.path.join(root_dir, file_relative_path)
        async with httpx.AsyncClient(follow_redirects=True, verify=CERTIFICATES_LOCATION) as client:
            response: httpx.Response = await client.send(request)
            if response.status_code == 200:
                logger.info(f"Downloaded {file_relative_path}")
                write_file_from_http_response(response, file_full_path)
                if callback:
                    callback(*callback_args)
                return SyncFileResult(True, project_id, file_relative_path, f"File {file_full_path} created")
            else:
                logger.error(f"Error while downloading {file_relative_path} from Projects API: {response.text}")
                return SyncFileResult(False, project_id, file_relative_path, f"Response {response.text}")
    except Exception as ex:
        logger.exception(f"Exception while Downloading {file_relative_path}: {ex}")
        return SyncFileResult(False, project_id, file_relative_path, f"Exception {ex}")


async def sync_push_delete_file(project_id: str, file_relative_path: str, user_credentials: Dict) -> SyncFileResult:
    """
    Deletes file from Projects API
    Args:
        project_id: id of the project
        file_relative_path: relative path of the file
        user_credentials: user credentials used to hit Projects API

    Returns: boolean

    """
    url = config["api"]["url"]
    url = f"{url}/projects/{project_id}/files/{quote(file_relative_path)}"
    request = httpx.Request(
        "DELETE",
        url,
        headers=get_headers(user_credentials),
    )

    async with httpx.AsyncClient(
        follow_redirects=True, timeout=FIVE_MINUTES_TIMEOUT, verify=CERTIFICATES_LOCATION
    ) as client:
        try:
            response: httpx.Response = await client.send(request)
            if response.status_code >= 200 and response.status_code < 300:
                return SyncFileResult(True, project_id, file_relative_path, f"File {file_relative_path} deleted")
            return SyncFileResult(False, project_id, file_relative_path, f"Response {response.text}")
        except httpx.TimeoutException as ex:
            logger.exception(f"Timeout while deleting {file_relative_path}: {ex}")
            return SyncFileResult(False, project_id, file_relative_path, f"Timeout {ex}")
        except Exception as ex:
            logger.exception(f"Exception while uploading {file_relative_path}: {ex}")
            return SyncFileResult(False, project_id, file_relative_path, f"Exception {ex}")


async def get_file_metadata(project_id: str, relative_file_path: str) -> Dict:
    """
    Get file metadata stored in the DB
    Args:
        project_id(str): project that te file belongs to
        relative_file_path(str): file relative path within the project
    Returns: boolean
    """
    if project := LocalProject.get(project_id):
        if file := project.get_content_by_key(relative_file_path):
            return file.metadata
    return {}


async def put_file(
    project_id: str, file_full_path: str, relative_file_path: str, user_credentials: Dict
) -> Optional[str]:
    """
    Writes a file to the storage, straight to S3. In this case the file is uploaded in a two-step process where
    first it hits Projects API in order to retrieve the S3 signed URL and then finally write the file
    to the S3.
    Writing the file straight to S3 eliminates the file size restriction which is set for Projects API.
    Args:
        project_id: id of the project
        file_full_path: file full path
        relative_file_path: file relative path
        user_credentials: user anaconda.cloud credentials (after migration: anaconda.com)

    Returns: string containing the file_version_id or None if the upload failed

    """

    file_content = read_file_content(file_full_path)

    try:
        uri = f"projects/{project_id}/file-preload/{quote(relative_file_path)}"
        metadata = await get_file_metadata(project_id, relative_file_path)
        response_projects_api_s3: httpx.Response = await cloud_request(
            uri, RequestMethods.PUT, user_credentials, json_payload={"metadata": metadata}
        )
        if response_projects_api_s3.status_code == 201:
            json_response = response_projects_api_s3.json()
            s3_signed_url = json_response["signed_url"]
            s3_request = httpx.Request("PUT", s3_signed_url, content=file_content)
            response_s3 = await make_request(s3_request, FIVE_MINUTES_TIMEOUT)
            if response_s3.status_code == 200:
                file_version_id = json_response["file_version_id"]
                commit_uri = f"projects/{project_id}/file-versions/{file_version_id}"
                response_commit: httpx.Response = await cloud_request(
                    commit_uri, RequestMethods.PATCH, user_credentials
                )
                if response_commit.status_code > 199 and response_commit.status_code < 300:
                    logger.debug("File commit OK")
                    return file_version_id
                logger.debug("File failed to commit")
                return None
            else:
                logger.error(f"S3 returned failed response: {response_s3}")
        logger.error(f"Error while putting file. Cloud request {response_projects_api_s3.content}")
        return None
    except httpx.TimeoutException as ex:
        logger.error(f"Timeout while writing file {relative_file_path}.{ex}")
        return None
    except httpx.NetworkError as ex:
        logger.error(f"Network error while writing file {relative_file_path}. {ex}")
        return None
    except Exception as ex:
        logger.error(f"Error while writing file {relative_file_path} {ex}")
        return None


async def sync_push_write_file(
    project_id: str,
    file_full_path: str,
    relative_file_path: str,
    user_credentials: Dict,
) -> UploadFileResult:
    """
    Uploads a local file (not a directory) to the Projects Service
    Args:
        project_id: Id of the project that contains the files
        file_full_path: Full local path to the file
        relative_file_path: Relative path to the file (relative to where the projects are stored locally)
        user_credentials: dict containing users credentials. Should have at least the access_token

    Returns: UploadFileResult indicating whether or not the operation succeeded

    """
    if not is_file(file_full_path):
        return UploadFileResult(
            False, project_id, relative_file_path, "Tried to upload an invalid file. Maybe it's a dir?"
        )

    project = LocalProject.get(project_id)

    try:
        project.set_content_sync_state(relative_file_path, LocalStorageSyncState.SYNCING)
        project.update(auto_commit=False)
        upload_start_time = time.time()
        if file_version_id := await put_file(
            project_id,
            file_full_path,
            relative_file_path,
            user_credentials,
        ):
            project.set_content_sync_state(relative_file_path, LocalStorageSyncState.SYNCED)
            project.update_file_version_id(relative_file_path, file_version_id)
            project.update(db)
            logger.info(f"File {file_full_path} uploaded")
            result = UploadFileResult(True, project.id, relative_file_path, f"{file_full_path} uploaded")
        else:
            project.set_content_sync_state(file_full_path, LocalStorageSyncState.SYNC_ERROR)
            project.update(db)
            logger.error(f"Error while uploading {file_full_path} to Projects API")
            result = UploadFileResult(False, project_id, relative_file_path, "Response")
        upload_elapsed_time = time.time() - upload_start_time
        write_elapsed_time(
            MetricNames.UPLOAD_ELAPSED_TIME,
            upload_elapsed_time,
            UPLOAD_FILE_LOG_FILE,
            f"{file_full_path} | {os.path.getsize(file_full_path) / (1024 * 1024)} MB",
        )
        return result
    except httpx.TimeoutException as ex:
        project.set_content_sync_state(relative_file_path, LocalStorageSyncState.SYNC_ERROR)
        project.update(db)
        logger.exception(f"Timeout while uploading {file_full_path}: {ex}")
        return UploadFileResult(False, project_id, relative_file_path, f"Exception {ex}")
    except Exception as ex:
        project.set_content_sync_state(relative_file_path, LocalStorageSyncState.SYNC_ERROR)
        project.update(db)
        logger.exception(f"Exception while uploading {file_full_path}: {ex}")
        return UploadFileResult(False, project_id, relative_file_path, f"Exception {ex}")


def sync_result_processor(sync_results: Dict, result_type) -> None:
    """
    Processes the synchronization results and updates the local project files' sync state accordingly.

    It updates the file status for all files in the local project files to 'SYNCED'.
    Finally, it iterates through the `sync_results` dictionary and updates the file
    status for any modified files.

    Args:
        sync_results (Dict): A dict containing the modified files to be synced.
        result_type: The expected type of result in `sync_results`. (SyncFileResult | UploadFileResult)

    Returns:
    """

    def _get_project_id(sync_results: Dict) -> str:
        """
        Returns the project ID from the first element in the sync_results dictionary.

        Args:
            sync_results (Dict): A dict containing the modified files to be synced.

        Returns:
            project_id, or raises an error if the input is invalid.
        """
        try:
            first_element = next(iter(sync_results.values()))
            return first_element.project_id
        except StopIteration:
            logger.warning("sync_results is empty, no files to be updated")
        except AttributeError:
            logger.warning("The first item in sync_results does not have a 'project_id' attribute")
        except Exception as e:
            logger.warning(f"An unexpected error occurred: {e}")

    if not sync_results:
        return None

    project = LocalProject.get(_get_project_id(sync_results))

    # This ensures that all files are marked as synced
    project_files = project.list_files()
    for project_file in project_files:
        project.set_content_sync_state(project_file.key, LocalStorageSyncState.SYNCED)

    # Now update the file status only for files that have been modified during this sync operation.
    for _, task_result in sync_results.items():
        if isinstance(task_result, result_type):
            sync_state = LocalStorageSyncState.SYNCED if task_result.success else LocalStorageSyncState.SYNC_ERROR
            project.set_content_sync_state(task_result.file_path, sync_state)

    project.update()
