"""OpenTelemetry Aleph Alpha instrumentation"""

import logging
import os
from typing import Collection, List, Optional, Union

from opentelemetry import context as context_api
from opentelemetry._logs import Logger, get_logger
from opentelemetry.instrumentation.alephalpha.config import Config
from opentelemetry.instrumentation.alephalpha.event_emitter import emit_event
from opentelemetry.instrumentation.alephalpha.event_models import (
    CompletionEvent,
    PromptEvent,
)
from opentelemetry.instrumentation.alephalpha.span_utils import (
    set_completion_attributes,
    set_prompt_attributes,
)
from opentelemetry.instrumentation.alephalpha.utils import dont_throw
from opentelemetry.instrumentation.alephalpha.version import __version__
from opentelemetry.instrumentation.instrumentor import BaseInstrumentor
from opentelemetry.instrumentation.utils import (
    _SUPPRESS_INSTRUMENTATION_KEY,
    unwrap,
)
from opentelemetry.semconv._incubating.attributes import (
    gen_ai_attributes as GenAIAttributes,
)
from opentelemetry.semconv_ai import (
    SUPPRESS_LANGUAGE_MODEL_INSTRUMENTATION_KEY,
    LLMRequestTypeValues,
    SpanAttributes,
)
from opentelemetry.trace import SpanKind, Tracer, get_tracer
from opentelemetry.trace.span import Span
from opentelemetry.trace.status import Status, StatusCode
from wrapt import wrap_function_wrapper

logger = logging.getLogger(__name__)

_instruments = ("aleph_alpha_client >= 7.1.0",)

WRAPPED_METHODS = [
    {
        "method": "complete",
        "span_name": "alephalpha.completion",
    },
]

TRACELOOP_TRACE_CONTENT = "TRACELOOP_TRACE_CONTENT"


def should_send_prompts():
    return (
        os.getenv(TRACELOOP_TRACE_CONTENT) or "true"
    ).lower() == "true" or context_api.get_value("override_enable_content_tracing")


def should_emit_events():
    return not Config.use_legacy_attributes


def _set_span_attribute(span, name, value):
    if value is not None:
        if value != "":
            span.set_attribute(name, value)
    return


@dont_throw
def _handle_message_event(
    event: PromptEvent, span: Span, event_logger: Optional[Logger], kwargs
):
    if span.is_recording():
        _set_span_attribute(span, GenAIAttributes.GEN_AI_REQUEST_MODEL, kwargs.get("model"))

    if should_emit_events():
        return emit_event(event, event_logger)
    else:
        return set_prompt_attributes(event, span)


def _handle_completion_event(event: CompletionEvent, span, event_logger, response):
    if span.is_recording():
        input_tokens = getattr(response, "num_tokens_prompt_total", 0)
        output_tokens = getattr(response, "num_tokens_generated", 0)

        _set_span_attribute(
            span, SpanAttributes.LLM_USAGE_TOTAL_TOKENS, input_tokens + output_tokens
        )
        _set_span_attribute(
            span, GenAIAttributes.GEN_AI_USAGE_OUTPUT_TOKENS, output_tokens
        )
        _set_span_attribute(span, GenAIAttributes.GEN_AI_USAGE_INPUT_TOKENS, input_tokens)

    if should_emit_events():
        emit_event(event, event_logger)
    else:
        set_completion_attributes(event, span)


def _with_tracer_wrapper(func):
    """Helper for providing tracer for wrapper functions."""

    def _with_tracer(tracer, event_logger, to_wrap):
        def wrapper(wrapped, instance, args, kwargs):
            return func(tracer, event_logger, to_wrap, wrapped, instance, args, kwargs)

        return wrapper

    return _with_tracer


def _llm_request_type_by_method(method_name):
    if method_name == "complete":
        return LLMRequestTypeValues.COMPLETION
    else:
        return LLMRequestTypeValues.UNKNOWN


def _parse_prompt_event(args, kwargs) -> PromptEvent:
    request = kwargs.get("request") if kwargs.get("request") else args[0]

    return PromptEvent(
        content=request.prompt.to_json(),
        role="user",
    )


def _parse_completion_event(response) -> List[CompletionEvent]:
    return CompletionEvent(
        index=0,
        message={
            "content": response.completions[0].completion,
            "role": "assistant",
        },
        finish_reason=response.completions[0].finish_reason,
    )


@_with_tracer_wrapper
def _wrap(
    tracer: Tracer,
    event_logger: Union[Logger, None],
    to_wrap,
    wrapped,
    instance,
    args,
    kwargs,
):
    """Instruments and calls every function defined in TO_WRAP."""
    if context_api.get_value(_SUPPRESS_INSTRUMENTATION_KEY) or context_api.get_value(
        SUPPRESS_LANGUAGE_MODEL_INSTRUMENTATION_KEY
    ):
        return wrapped(*args, **kwargs)

    name = to_wrap.get("span_name")
    llm_request_type = _llm_request_type_by_method(to_wrap.get("method"))
    span = tracer.start_span(
        name,
        kind=SpanKind.CLIENT,
        attributes={
            GenAIAttributes.GEN_AI_SYSTEM: "AlephAlpha",
            SpanAttributes.LLM_REQUEST_TYPE: llm_request_type.value,
        },
    )
    input_event = _parse_prompt_event(args, kwargs)
    _handle_message_event(input_event, span, event_logger, kwargs)

    response = wrapped(*args, **kwargs)

    if response:
        response_event = _parse_completion_event(response)
        _handle_completion_event(response_event, span, event_logger, response)
        if span.is_recording():
            span.set_status(Status(StatusCode.OK))

    span.end()
    return response


class AlephAlphaInstrumentor(BaseInstrumentor):
    """An instrumentor for Aleph Alpha's client library."""

    def __init__(self, exception_logger=None, use_legacy_attributes=True):
        super().__init__()
        Config.exception_logger = exception_logger
        Config.use_legacy_attributes = use_legacy_attributes

    def instrumentation_dependencies(self) -> Collection[str]:
        return _instruments

    def _instrument(self, **kwargs):
        tracer_provider = kwargs.get("tracer_provider")
        tracer = get_tracer(__name__, __version__, tracer_provider)

        event_logger = None

        if should_emit_events():
            logger_provider = kwargs.get("logger_provider")
            event_logger = get_logger(
                __name__,
                __version__,
                logger_provider=logger_provider,
            )

        for wrapped_method in WRAPPED_METHODS:
            wrap_method = wrapped_method.get("method")
            wrap_function_wrapper(
                "aleph_alpha_client",
                f"Client.{wrap_method}",
                _wrap(tracer, event_logger, wrapped_method),
            )

    def _uninstrument(self, **kwargs):
        for wrapped_method in WRAPPED_METHODS:
            unwrap(
                "aleph_alpha_client.Client",
                wrapped_method.get("method"),
            )
