#!/bin/sh
# SPDX-License-Identifier: MIT

#
# This script is responsible for loading firmware blobs for the HexagonFS
# on qcom devices. It assumes to be run after msm-firmware-loader.
#

# Base directory that msm-firmware-loader uses.
BASEDIR="/lib/firmware/msm-firmware-loader"

# HexagonFS base directory.
HEXAGONFSDIR="/usr/share/qcom"

# socinfo file base directory.
SOCINFODIR="/usr/share/hexagonfs-firmware-loader"

# Preparations:

mount -o mode=755,nodev,noexec,nosuid -t tmpfs none "$HEXAGONFSDIR"

mkdir "$HEXAGONFSDIR/sensors"

# Get the slot suffix for A/B devices.
# If qbootctl is available query the active slot using that, otherwise rely on
# the kernel cmdline to contain the slot suffix.
# On non-A/B devices the return value will be empty.
# https://source.android.com/docs/core/architecture/bootloader/updating#slots
ab_get_slot() {
	if command -v qbootctl > /dev/null; then
		ab_slot_suffix=$(qbootctl -a | grep -o 'Active slot: _[ab]' | cut -d ":" -f2 | xargs) || :
	else
		ab_slot_suffix=$(grep -o 'androidboot\.slot_suffix=..' /proc/cmdline |  cut -d "=" -f2) || :
	fi
	echo "$ab_slot_suffix"
}

# Get the device codename (in the form of vendor-device) from the compatible
# in the devicetree.
DEVICE=$(tr '\0,' '\n-' < /sys/firmware/devicetree/base/compatible | head -n1)

# Symlink the files and directories from msm-firmware-loader mounts:

ln -s "$BASEDIR/mnt/vendor$(ab_get_slot)/etc/acdbdata" "$HEXAGONFSDIR"/acdb

# Some devices have a dsp partition, others have /vendor/dsp.
# The ones with dsp partition have a mount point for it at /vendor/dsp, but
# we don't mount it there, so we need to fall back to /vendor/dsp.
if [ -d "$BASEDIR/mnt/dsp$(ab_get_slot)" ]
then
	ln -s "$BASEDIR/mnt/dsp$(ab_get_slot)" "$HEXAGONFSDIR"/dsp
else
	ln -s "$BASEDIR/mnt/vendor$(ab_get_slot)/dsp" "$HEXAGONFSDIR"/dsp
fi

ln -s "$BASEDIR/mnt/vendor$(ab_get_slot)/etc/sensors/config" "$HEXAGONFSDIR"/sensors/config

# The permissions on persist are generally 600 and ownership is uid 1000.
# This means the fastrpc user cannot read it if we symlink - so we need to copy.
cp -r "$BASEDIR"/mnt/persist/sensors/registry/registry "$HEXAGONFSDIR"/sensors/registry
chmod 644 "$HEXAGONFSDIR"/sensors/registry/*

ln -s "$BASEDIR/mnt/vendor$(ab_get_slot)/etc/sensors/sns_reg_config" "$HEXAGONFSDIR"/sensors/sns_reg.conf

# Some devices (hey, Xiaomi!) require a bunch of files from /sys/devices/soc0.
# Since that is device-specific, let's check if we have a directory for this device.
if [ -d "$SOCINFODIR"/"$DEVICE" ]
then
	ln -s "$SOCINFODIR"/"$DEVICE" "$HEXAGONFSDIR"/socinfo
fi
